#!/usr/local/bin/perl
use lib '/usr/local/rib/lib';
# NHSE Repository in a Box (RIB)
#
# The authors of this software are Paul McMahan and Jeff Horner.
# Copyright (c) 1997 by the University of Tennessee.
# Permission to use, copy, modify, and distribute this software for any
# purpose without fee is hereby granted, provided that this entire notice
# is included in all copies of any software which is or includes a copy
# or modification of this software and in all copies of the supporting
# documentation for such software.
# THIS SOFTWARE IS BEING PROVIDED "AS IS", WITHOUT ANY EXPRESS OR IMPLIED
# WARRANTY.  IN PARTICULAR, NEITHER THE AUTHORS NOR UNIVERSITY OF TENNESSEE
# MAKE ANY REPRESENTATION OR WARRANTY OF ANY KIND CONCERNING THE
# MERCHANTABILITY OF THIS SOFTWARE OR ITS FITNESS FOR ANY PARTICULAR PURPOSE.
#
use strict;
use integer;
use FileHandle; 
use RIB::Util ();
use RIB::DomainParser ();
use RIB::ConfigParser ();
use RIB::BIDMParser ();
use Data::Dumper ();

use vars qw($DEBUG);
#$DEBUG = 1;
################# Acquire Important Variables and Arguments

my $util = RIB::Util->new();
my ($repository,$buf);
$buf = '';
my $RIBDIR = $util->GetRibDir();
my $RIBURL = $util->GetRibUrl();
my $RIBSP = $util->GetRibIsearchCgiPath();
my $RIBI = $util->GetRibIindexPath();
if (@ARGV){
    $repository = $ARGV[0];
} else {
    $repository = $util->GetRepoName();
}
unless ($repository) {
    $util->ErrorMessage("A repository was not specified in your input");
}
my $filepath = "$RIBDIR/repositories/$repository";

select(STDERR) ; $| = 1; # no output buffering
select(STDOUT); $| = 1;


################# Print Header

$util->PrintHeader();

print "<html>\n";
print "<head><title>Generating deployment grid</title></head>\n";
print "<body bgcolor=#FFFFF0>\n";
print "<center><h1>Generating deployment grid</h1></center><P><hr><P>\n";

################ Load Configuration Object

my $cp = RIB::ConfigParser->new();
unless ($cp->load_config("$filepath/conf/BIDM.conf")){
    print "</pre>\n";
    $util->HtmlCroak("There is a problem with $repository\'s ".
	"configuration file. Problem: " . $cp->ErrorMsg() . ". Please ".
	"contact your RIB administrator");
}


#################
my %assets = {};         #assets in ncsa_teamA repos and their catalog pages
my %machines = {};       #machines in ncsa_teamA_deployment repos and their catalog pages
my %deployments= {};     #deployments in ncsa_teamA_deployment repos and their statuses


################# load %machines
unless (opendir (DIR,"$filepath/objects/Machine")) {
    print "</pre>\n";
    $util->HtmlCroak("Couldn't open the directory "
                   . "$filepath/objects/Machine."
                   . "<p>Reason: $!<p>Please contact rib\@nhse.org");
}
foreach (grep /\.html?$/i , readdir(DIR) ){
  my $file = "$filepath/objects/Machine/$_";

  # parse the deployment file
  my $Machine = RIB::BIDMParser->new();
  $Machine->parse_file($file);

  # determine the catalog entry for this asset
  my $ce = $file;
  $ce =~ s/$RIBDIR/$RIBURL/i;
  $ce =~ s/objects/catalog/i;

  my $machine = $Machine->valueof("Name");
  $machines{$machine} = $ce;
}
close DIR;

################# load %assets
unless (opendir (DIR,"$RIBDIR/repositories/ncsa_teamA/objects/Asset")) {
    print "</pre>\n";
    $util->HtmlCroak("Couldn't open the directory "
                   . "$RIBDIR/repositories/ncsa_teamA/objects/Asset"
                   . "<p>Reason: $!<p>Please contact rib\@nhse.org");
}
foreach (grep /\.html?$/i , readdir(DIR) ){
  my $file = "$RIBDIR/repositories/ncsa_teamA/objects/Asset/$_";

  # parse the deployment file
  my $Asset = RIB::BIDMParser->new();
  $Asset->parse_file($file);

  # determine the catalog entry for this asset
  my $ce = $file;
  $ce =~ s/$RIBDIR/$RIBURL/i;
  $ce =~ s/objects/catalog/i;

  my $asset = $Asset->valueof("Name");
  $assets{$asset} = $ce;
}
close DIR;


################# Get Deployment objects
unless (opendir (DIR,"$filepath/objects/Deployment")) {
    print "</pre>\n";
    $util->HtmlCroak("Couldn't open the directory "
                   . "$filepath/objects/Deployment."
                   . "<p>Reason: $!<p>Please contact rib\@nhse.org");
}
foreach ( grep /\.html?$/i , readdir(DIR) ){
  # parse the deployment file
  my $Deployment = RIB::BIDMParser->new();
  $Deployment->parse_file("$filepath/objects/Deployment/$_");

  ######################################################
  # determine which Machine this deployment is on
  my $file1 = $Deployment->valueof("DeployedOn");
  $file1 =~ s/$RIBURL/$RIBDIR/i;
  my $DeployedOn = RIB::BIDMParser->new();
  $DeployedOn->parse_file($file1);
  my $machine = $DeployedOn->valueof("Name");

  ######################################################
  # determine which Asset this is a deployment of
  my $file2= $Deployment->valueof("IsDeploymentOf");
  $file2 =~ s/$RIBURL/$RIBDIR/i;
  my $IsDeploymentOf = RIB::BIDMParser->new();
  $IsDeploymentOf->parse_file($file2);
  my $asset = $IsDeploymentOf->valueof("Name");

  ######################################################
  # now store the status of the deployment and url to its catalog entry
  my $status = $Deployment->valueof("Status");
  $deployments{$machine}{$asset} =
     "<a href=\"$RIBURL/repositories/$repository/catalog/Deployment/$_\">$status</a>";
}
closedir DIR;


##################################################################
# get lock for the grid html file and then open it
my $grid = "$filepath/catalog/grid.html";
my $gridlock = "$filepath/catalog/grid.html.lock";
while (-e $gridlock) {
  if ((stat($gridlock))[10] < (time - 60)) {
    unlink $gridlock or $util->HtmlCroak("Can't unlink grid lock file : $!");
  }
  sleep 5;
}
unless (open(GRIDLOCK, ">$gridlock")) {
  $util->HtmlCroak("Can't create grid lock file : $!");
}
close GRIDLOCK;

unless (open (GRID,">$grid")) {
  unlink ($gridlock);
  $util->HtmlCroak("Can't create grid file : $!");
}


##################################################################
# print the grid
select GRID;

print << 'EOF';
<HTML>
<HEAD><TITLE> NCSA Team A Deployment Grid</TITLE></HEAD>

<BODY TEXT="#000000" ALINK="#FFFFFF" BGCOLOR="#666666">
<TABLE BORDER="0" CELLSPACING="0" CELLPADDING="0" WIDTH="800">

  <TR VALIGN=BOTTOM>
    <TD COLSPAN="5"><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></TD>
    <TD COLSPAN="2" ROWSPAN="3" ALIGN=LEFT><IMG src="/rib/images/alliance/cornerlogo.gif" WIDTH="111" HEIGHT="105" BORDER=0 ALT="[logo]"></TD>
  </TR>

  <TR VALIGN=TOP>
    <TD><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></TD>
    <TD ALIGN=LEFT COLSPAN="2"><IMG src="/rib/images/alliance/header-left.gif" WIDTH="45" HEIGHT="60" BORDER=0 ALT=" "></TD>
    <TD ALIGN=LEFT BGCOLOR="#CCCCCC"><IMG src="/rib/images/alliance/header-alliance.gif" WIDTH="282" HEIGHT="60" BORDER=0 ALT="National Computational Science Alliance"></TD>
    <TD ALIGN=RIGHT><IMG src="/rib/images/alliance/header-right.gif" WIDTH="14" HEIGHT="60" BORDER=0 ALT=" "></TD>
  </TR>

  <TR VALIGN=TOP>
    <TD><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></TD>
    <TD BGCOLOR="#CCCCCC"><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></TD>
    <TD BGCOLOR="#FFFFFF"><IMG src="/rib/images/alliance/leftmargin-top.gif" WIDTH="25" HEIGHT="20" ALT=" "></TD>
    <TD BGCOLOR="#FFFFFF" valign=bottom align=center><IMG SRC="/rib/images/alliance/ItTakesAnAlliance.gif" WIDTH=206 HEIGHT=16 BORDER=0 VSPACE=0 ALT="It Takes An Alliance To Build A Grid"></TD>
    <TD BGCOLOR="#FFFFFF"><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></TD>
  </TR>

  <tr ALIGN=RIGHT> 
    <TD><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></TD>
    <td BGCOLOR="#CCCCCC"><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></td>
    <td colspan=4 BGCOLOR="#FFFFFF"><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></td>
    <TD><img src="/rib/images/alliance/rightmargin-top.gif"></TD> 
  </tr>

  <tr> 
    <TD><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></TD>
    <td BGCOLOR="#CCCCCC"><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></td>
    <td BGCOLOR="#FFFFFF"><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></td>
    <td BGCOLOR="#FFFFFF">
EOF

print "<center><h2>Software Deployment Grid for NCSA Team A</h2></center>\n";
print "<p><hr><P>\n";
print "Shown below is a table which describes the current status of NCSA Team A\n";
print "software deployment as of ", `date '+%D'`, "<P>\n";
print "Clicking on a machine name along the top row will show a description of\n";
print "that machine. Clicking\n";
print "on a software title along the left hand column will show a description of\n";
print "that software. And clicking on\n";
print "the deployment status inside the table will give details about that deployment.\n";
print "<P>Other items of interest for the deployment team are the Team A ";
print "<a href=../forum/index.html>deployment discussion forum</a>, the Team A ";
print "<a href=../../ncsa_teamA/catalog/>software catalog</a>\n";
print "and the <a href=http://alliance.ncsa.uiuc.edu/>Alliance Home Page</a>.\n";
print "<P><HR><P>\n";
print "<center><table border=5 cellpadding=10>\n";
print "<tr><th>&nbsp;</th>";
my $machine;
foreach $machine (sort keys %machines) {
  print "<th align=center><a href=" . $machines{$machine} . ">$machine</a></th>\n";
}
print "</tr>\n";

my $asset;
foreach $asset (sort keys %assets) {
  print "<tr>\n";
  print "<th align=center><a href=" . $assets{$asset} . ">$asset</a></th>\n";
  foreach $machine (sort keys %machines) {
    print "<td align=center>";
    if ($deployments{$machine}{$asset}) {
      print $deployments{$machine}{$asset};
    } else {
      print "-";
    }
    print "</td>\n";
  }
  print "</tr>\n";
}
print "</table></center>\n";

print << 'EOF';
    </td>
    <td BGCOLOR="#FFFFFF"><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></td>
    <td BGCOLOR="#FFFFFF"><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></td>
    <TD BGCOLOR="#CCCCCC"><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></TD> 
  </tr>

  <TR VALIGN=BOTTOM>
    <TD WIDTH="112" NOWRAP ROWSPAN=2><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></TD>
    <TD WIDTH="20" NOWRAP ROWSPAN=2 BGCOLOR="#CCCCCC"><IMG src="/rib/images/alliance/footer-left1.gif" WIDTH=20 HEIGHT=35 ALT=" "></TD>
    <TD WIDTH="25" NOWRAP ROWSPAN=2 BGCOLOR="#FFFFFF"><IMG src="/rib/images/alliance/footer-left2.gif" WIDTH=25 HEIGHT=35 ALT=" "></TD>
    <TD WIDTH="100%" BGCOLOR="#FFFFFF">
      <FONT FACE="arial,helvetica" SIZE="1"> 
        <A HREF="mailto:alliance-webmaster@ncsa.uiuc.edu">teama@nhse.org</A><BR>
      </FONT><BR>
    </TD>
    <TD WIDTH="14" NOWRAP BGCOLOR="#FFFFFF"><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH="1" HEIGHT="25" ALT=" "></TD>
    <TD WIDTH="22" NOWRAP ROWSPAN=2 BGCOLOR="#FFFFFF"><IMG src="/rib/images/alliance/footer-right1.gif" WIDTH=22 HEIGHT=35 ALT=" "></TD>
    <TD WIDTH="89" NOWRAP ROWSPAN=2 BGCOLOR="#CCCCCC"><IMG src="/rib/images/alliance/footer-right2.gif" WIDTH=89 HEIGHT=81 ALT=" "></TD>
  </TR>

  <TR VALIGN=TOP>
    <TD BGCOLOR="#CCCCCC"><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH=1 HEIGHT=2 ALT=" " VSPACE=0></TD>
    <TD BGCOLOR="#CCCCCC"><IMG src="/rib/images/alliance/dot_clear.gif" WIDTH=1 HEIGHT=2 ALT=" " VSPACE=0></TD>
  </TR>
</TABLE>
</BODY>
</HTML>
EOF

close GRID;
unlink($gridlock);

##################################################################
# report success
select STDOUT;
print "<P>Grid Generation was successful.<P>\n";
print "Click <a href=/rib/repositories/ncsa_teamA_deployment/catalog/grid.html>here</a>";
print " to see the grid.\n";
exit(0);
