#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: ex3.c,v 1.7 1998/03/06 00:17:45 bsmith Exp $";
#endif
/*
       This program uses PVODE routines to solve:

          PDE: U_t = U*U_x + epsilon*U_xx 

               U(0,x)=Sin(2*pi*x)
               U(t,0)=U(t,1)

       by the method of lines.  Approximated by:
 
          F(t,u) = u_i*(u_i+1-u_i-1)/(2h)+epsilon*(u_i+1 - 2u_i + u_i-1)/h^2

*/

static char help[] = "Solves 1D viscus Burgers' equation.\n\n";

#include "gvec.h"
#include "sys.h"
#include "ts.h"
#include <math.h>

/*
    Define a data structure to store all the application specific information
  about the problem.
*/
typedef struct {
  GVec   localwork1,localwork2; /* location for local work (with ghost points) vectors */
  DA     da;                    /* manages ghost point communication */
  Viewer viewer;
  int    M;                     /* total number of grid points */
  double h;                     /* mesh width h = 1/(M-1) */
  int    nox;                   /* indicates problem is to be run without graphics */ 
  double epsilon;	        /* the viscus coefficient */
} AppCtx;

extern int Monitor(TS, int, double , Vec, void *);
extern int RHSFunction(TS,double,Vec,Vec,void*);
extern int Initial(Vec, void*);
extern int RHSJacobian(TS,double,Vec,Mat*,Mat*,MatStructure *,void*);

extern PointFunction InitialCondition;

int main(int argc,char **argv)
{
  int           ierr,  time_steps = 2000, steps, flg, size;
  AppCtx        appctx;
  GVec          global;
  double        dt,ftime;
  TS            ts;
  Mat           A = 0;
  Draw          draw;
  Viewer        viewer;
  char          tsinfo[120];
  Grid          grid;
  GridData      griddata;
  DA            da;
  MatFDColoring fd = 0;
  ISColoring    coloring;
 
  PetscInitialize(&argc,&argv,(char*)0,help);
  ierr = GVecInitialize();  CHKERRA(ierr);
  MPI_Comm_size(PETSC_COMM_WORLD,&size);

  appctx.M = 60;
  ierr = OptionsGetInt(PETSC_NULL,"-M",&appctx.M,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-time",&time_steps,&flg);CHKERRA(ierr);
    
  ierr = OptionsHasName(PETSC_NULL,"-nox",&flg);CHKERRA(ierr); 
  if (flg) appctx.nox = 1; else appctx.nox = 0;

  ierr = GridCreateRectangular1D(PETSC_COMM_WORLD,appctx.M,0.0,1.0,PETSC_NULL,
                                 DISCRETIZATION_LINEAR,&grid); CHKERRA(ierr);
  ierr = GridRectangularSetPeriodicType(grid,DA_XPERIODIC); CHKERRA(ierr);
  ierr = GridSetUp(grid); CHKERRA(ierr);

/*  remove next 3 lines eventually */
  ierr = GridGetGridData(grid,&griddata); CHKERRA(ierr);
  ierr = GridDataRectangularGetDA(griddata,&appctx.da); CHKERRA(ierr);
  
  /* make parallel vector for containing the solution */
  ierr = GridCreateGVec(grid,&global); CHKERRA(ierr);

  /* make local work vectors (including ghost points) for evaluating right hand side function */
  ierr = GVecGetLocalWorkGVec(global,&appctx.localwork1); CHKERRA(ierr);
  ierr = GVecGetLocalWorkGVec(global,&appctx.localwork2); CHKERRA(ierr);

  /* Set up display to show solution */

  ierr = ViewerDrawOpenX(PETSC_COMM_WORLD,0,"",80,380,400,160,&appctx.viewer);CHKERRA(ierr);
  ierr = ViewerDrawGetDraw(appctx.viewer,&draw); CHKERRA(ierr);
  ierr = DrawSetDoubleBuffer(draw); CHKERRA(ierr);   

  appctx.h = 1.0/(appctx.M-1.0);
  appctx.epsilon = 0.01;

  /* set initial conditions */
  GVecEvaluateFunction(global,InitialCondition,&appctx);
  /*
  ierr = Initial(global,&appctx); CHKERRA(ierr); */
 
  /* make timestep context */
  ierr = TSCreate(PETSC_COMM_WORLD,TS_NONLINEAR,&ts); CHKERRA(ierr);
  ierr = TSSetType(ts,TS_PVODE); CHKERRA(ierr);
  ierr = TSSetMonitor(ts,Monitor,&appctx); CHKERRA(ierr);

  dt = 0.1;

  /*
     The user provides the RHS and Jacobian
  */
  ierr = TSSetRHSFunction(ts,RHSFunction,&appctx); CHKERRA(ierr);

  ierr = OptionsHasName(PETSC_NULL,"-mat_fd",&flg); CHKERRA(ierr);
  if (flg) {
    ierr = GridDataRectangularGetDA(griddata,&da);CHKERRA(ierr);
    ierr = DAGetColoring(da,&coloring,&A);CHKERRA(ierr);
    ierr = MatFDColoringCreate(A,coloring,&fd);CHKERRA(ierr);
    ierr = MatFDColoringSetFunction(fd,(int (*)(void))RHSFunction,&appctx); CHKERRA(ierr);
    ierr = ISColoringDestroy(coloring);CHKERRA(ierr);
    ierr = TSSetRHSJacobian(ts,A,A,TSDefaultComputeJacobianWithColoring,fd); CHKERRA(ierr);  
  } else {
    ierr = GridCreateGMat(grid,&A); CHKERRA(ierr);
    ierr = TSSetRHSJacobian(ts,A,A,RHSJacobian,&appctx); CHKERRA(ierr);  
  }

  ierr = TSSetInitialTimeStep(ts,0.0,dt); CHKERRA(ierr);
  ierr = TSSetDuration(ts,time_steps,10.0); CHKERRA(ierr);
  ierr = TSSetSolution(ts,global); CHKERRA(ierr);
  ierr = TSSetFromOptions(ts);CHKERRA(ierr);


  ierr = TSSetUp(ts); CHKERRA(ierr);
  ierr = TSStep(ts,&steps,&ftime); CHKERRA(ierr);
  ierr = ViewerStringOpen(PETSC_COMM_WORLD,tsinfo,120,&viewer); CHKERRA(ierr);
  ierr = TSView(ts,viewer); CHKERRA(ierr);

  PetscPrintf(PETSC_COMM_WORLD,"Done %s \n",tsinfo);

  ierr = ViewerDestroy(viewer); CHKERRA(ierr);
  ierr = TSDestroy(ts); CHKERRA(ierr);
  ierr = ViewerDestroy(appctx.viewer); CHKERRA(ierr);
  ierr = VecDestroy(appctx.localwork1); CHKERRA(ierr);
  ierr = VecDestroy(appctx.localwork2); CHKERRA(ierr);
  ierr = VecDestroy(global); CHKERRA(ierr);
  ierr = GridDestroy(grid); CHKERRA(ierr);
  if (fd) {ierr = MatFDColoringDestroy(fd); CHKERRA(ierr); }

  ierr= MatDestroy(A); CHKERRA(ierr);

  PetscFinalize();
  return 0;
}

/* -------------------------------------------------------------------*/

int InitialCondition(int n,double *x,double *y,double *z,Scalar *result,void *ctx)
{
  int i;
  for (i=0; i<n; i++ ) {
    result[i] = sin(PETSC_PI*x[i]);
  }
  return 0;
}


/* the monitor: show the intermedia results graphically */
int Monitor(TS ts, int step, double time,Vec global, void *ctx)
{
  AppCtx   *appctx = (AppCtx*) ctx;
  int      ierr;
  MPI_Comm comm;

  ierr = PetscObjectGetComm((PetscObject)ts,&comm); CHKERRQ(ierr);

  ierr = GVecView(global,appctx->viewer); CHKERRQ(ierr);

  PetscPrintf(comm,"timestep %d time %g ... done! \n",step,time);

  return 0;
}


int RHSFunction(TS ts, double t,Vec globalin, Vec globalout, void *ctx)
{
  AppCtx *appctx = (AppCtx*) ctx;
  DA     da = appctx->da;
  Vec    localwork2 = appctx->localwork2, localwork1 = appctx->localwork1;
  int    ierr,i,localsize; 
  Scalar *copyptr, *localptr,sc1,sc2;
  double epsilon = appctx->epsilon;
  double h = appctx->h;

  /*Extract local array */ 
  ierr = DAGlobalToLocalBegin(da,globalin,INSERT_VALUES,localwork2); CHKERRQ(ierr);
  ierr = DAGlobalToLocalEnd(da,globalin,INSERT_VALUES,localwork2); CHKERRQ(ierr);
  ierr = VecGetArray(localwork2,&localptr); CHKERRQ(ierr);

  /* Extract work vector */
  ierr = VecGetArray(localwork1,&copyptr); CHKERRQ(ierr);

  /* Update Locally - Make array of new values */
  /* Note: For the first and last entry I copy the value */
  /* if this is an interior node it is irrelevant */
  sc1 = 0.5/h;
  sc2 = epsilon/(h*h);
  ierr = VecGetLocalSize(localwork2,&localsize); CHKERRQ(ierr);
  copyptr[0] = localptr[0];
  for (i=1; i<localsize-1; i++) {
    copyptr[i] = sc1*localptr[i]*(localptr[i+1]-localptr[i-1])+
      sc2*(localptr[i+1] + localptr[i-1] - 2.0*localptr[i]);
  }
  copyptr[localsize-1] = localptr[localsize-1];
  ierr = VecRestoreArray(localwork1,&copyptr); CHKERRQ(ierr);

  /* Local to Global */
  ierr = DALocalToGlobal(da,localwork1,INSERT_VALUES,globalout); CHKERRQ(ierr);
  return 0;
}

int RHSJacobian(TS ts,double t,Vec globalin, Mat *AA, Mat *BB, MatStructure *str,void *ctx)
{
  Mat        A = *AA;
  AppCtx     *appctx = (AppCtx*) ctx;
  DA         da = appctx->da;
  Scalar     sc1,sc2, *localptr;
  double     epsilon = appctx->epsilon;
  double     h = appctx->h;
  Vec        localwork2 = appctx->localwork2;
  int        ierr,i,localsize, idx[3];
  Scalar     v[3];

  *str = SAME_NONZERO_PATTERN;

  /*Extract local array */ 
  ierr = DAGlobalToLocalBegin(da,globalin,INSERT_VALUES,localwork2); CHKERRQ(ierr);
  ierr = DAGlobalToLocalEnd(da,globalin,INSERT_VALUES,localwork2); CHKERRQ(ierr);
  ierr = VecGetArray(localwork2,&localptr); CHKERRQ(ierr);


  sc1 = 0.5/h;    
  sc2 = epsilon/(h*h);
  
  ierr = VecGetLocalSize(localwork2,&localsize); CHKERRQ(ierr);

  /*
     Construct matrice one row at a time
  */
  for ( i=1; i<localsize-1; i++ ) {
    idx[0] = i-1; idx[1] = i; idx[2] = i+1;
    v[0] = sc1*localptr[i]+sc2;
    v[1] = sc1*localptr[i-1]-2.0*sc2+sc1*localptr[i+1];
    v[2] = sc1*localptr[i]+sc2;
    ierr = MatSetValuesLocal(A,1,&i,3,idx,v,INSERT_VALUES); CHKERRQ(ierr);
  }
  ierr = MatAssemblyBegin(A,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  ierr = MatAssemblyEnd(A,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);

  ierr = VecRestoreArray(localwork2,&localptr);

  return 0;
}






