#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: ex1.c,v 1.9 1997/11/28 16:21:52 bsmith Exp $";
#endif

static char help[] = "Demonstrates use of the SNES package to solve a\n\
system of nonlinear equations (uniprocessor).  These examples are taken\n\
from the MINPACK-2 test suite (which uses dense storage of the Jacobian\n\
matrix).  The command line options are:\n\
 -p <problem_number>, where the problems and their options are\n\
   1: Solid Fuel Ignition (dsfi)\n\
      -par <param>, where param is the Bratu parameter (0 <= par <= 6.81)\n\
      -mx <xg>, where xg = number of grid points in x-direction\n\
      -my <yg>, where yg = number of grid points in y-direction\n\
   2: Flow in a Driven Cavity (dfdc)\n\
      -par <param>, where param is the Reynolds number (par > 0)\n\
      -mx <xg>, where xg = number of grid points in x-direction\n\
      -my <yg>, where yg = number of grid points in y-direction\n\
   3: Flow in a Channel (dfic)\n\
      -par <param>, where param is the Reynolds number (par > 0)\n\
      -mx <xg>, where xg = number of grid points on the problem domain\n\
         Note:  xg must be a multiple of 8.\n\
   4: Swirling Flow between Disks (dsfd)\n\
      -par <param>, where param is the viscosity (par > 0)\n\
      -mx <xg>, where xg = number of grid points on the problem domain\n\
         Note:  xg must be a multiple of 14.\n\
   5: Human Heart Dipole (dhhd)\n\
      -hp <num>, where num indicates the problem variant (1 <= num <= 5)\n\
      Note:  The problem dimension is 8 for all variants.\n\
   6: Compustion of Propane - Reduced Formulation (dcpr)\n\
      Note:  The problem dimension is 5; no options.\n\
   7: Compustion of Propane - Full Formulation (dcpf)\n\
      Note:  The problem dimension is 11; no options.\n\n";

/* ------------------------------------------------------------------- */
/* We thank Brett Averick and Jorge More' for these test problems from */
/* the MINPACK-2 test suite.                                           */ 
/* ------------------------------------------------------------------- */

#if !defined(USE_PETSC_COMPLEX)

#include "petsc.h"
#include "snes.h"

/* User-defined application context */
typedef struct {
   int     mx;           /* discretization in x-direction */
   int     my;           /* discretization in y-direction */
   int     n;            /* problem dimension */
   int     problem;      /* test problem number */
   double  param;        /* test problem parameter */
   char    HHVariant[6]; /* char string for human heart problem */
} AppCtx;

int FormJacobian(SNES,Vec,Mat*,Mat*,MatStructure*,void*);
int FormFunction(SNES,Vec,Vec,void*);
int FormInitialGuess(AppCtx*,Vec);
int UserSetProblem(AppCtx*,int*,char**);

#if defined(HAVE_FORTRAN_CAPS)
#define dficfj_ DFICFJ
#define dsfdfj_ DSFDFJ
#define dhhdfj_ DHHDFJ
#define dcprfj_ DCPRFJ
#define dcpffj_ DCPFFJ
#define dsfifj_ DSFIFJ
#define dfdcfj_ DFDCFJ

#elif !defined(HAVE_FORTRAN_UNDERSCORE)
#define dficfj_ dficfj
#define dsfdfj_ dsfdfj
#define dhhdfj_ dhhdfj
#define dcprfj_ dcprfj
#define dcpffj_ dcpffj
#define dsfifj_ dsfifj
#define dfdcfj_ dfdcfj
#endif

#if defined(__cplusplus)
extern "C" {
#endif
void dsfifj_(int*,int*,Scalar*,Scalar*,Scalar*,int*,char*,Scalar*,int);
void dfdcfj_(int*,int*,Scalar*,Scalar*,Scalar*,int*,char*,Scalar*,int);
void dficfj_(int*,Scalar*,Scalar*,Scalar*,int*,char*,Scalar*,int*,int);
void dsfdfj_(int*,Scalar*,Scalar*,Scalar*,int*,char*,Scalar*,int*,int);
void dhhdfj_(int*,Scalar*,Scalar*,Scalar*,int*,char*,char*,int,int);
void dcprfj_(int*,Scalar*,Scalar*,Scalar*,int*,char*,int);
void dcpffj_(int*,Scalar*,Scalar*,Scalar*,int*,char*,int);
#if defined(__cplusplus)
}
#endif

int main(int argc,char **argv)
{
  SNES     snes;          /* SNES context */
  Vec      x, f;          /* solution, function vectors */
  Mat      J;             /* Jacobian matrix */
  AppCtx   user;          /* application context */
  SLES     sles;          /* SLES context */
  KSP      ksp;           /* Krylov solver context */
  PC       pc;            /* preconditioner context */
  int      ierr, its, nfails;

  PetscInitialize(&argc,&argv,(char *)0,help);

  /* Initialize test problem */
  ierr = UserSetProblem(&user,&argc,argv); CHKERRQ(ierr);

  /* Create vectors */
  ierr = VecCreate(PETSC_COMM_SELF,PETSC_DECIDE,user.n,&x); CHKERRA(ierr);
  ierr = VecDuplicate(x,&f); CHKERRA(ierr);

  /* Create Jacobian matrix context.  Note that we must use a dense
     sequential matrix because the MINPACK-2 codes require this format. */
  ierr = MatCreateSeqDense(PETSC_COMM_SELF,user.n,user.n,PETSC_NULL,&J); CHKERRA(ierr);

  /* Create nonlinear solver */
  ierr = SNESCreate(PETSC_COMM_SELF,SNES_NONLINEAR_EQUATIONS,&snes); CHKERRA(ierr);

  /* Set function and Jacobian evaluation routines */
  ierr = SNESSetFunction(snes,f,FormFunction,(void *)&user); CHKERRA(ierr);
  ierr = SNESSetJacobian(snes,J,J,FormJacobian,(void *)&user); CHKERRA(ierr);

  /* Set default linear solver (LU); this can be overridden at runtime */
  ierr = SNESGetSLES(snes,&sles); CHKERRA(ierr);
  ierr = SLESGetKSP(sles,&ksp); CHKERRA(ierr);
  ierr = SLESGetPC(sles,&pc); CHKERRA(ierr);
  ierr = KSPSetType(ksp,KSPPREONLY); CHKERRA(ierr);
  ierr = PCSetType(pc,PCLU); CHKERRA(ierr);

  /* Set runtime options */
  ierr = SNESSetFromOptions(snes); CHKERRA(ierr);

  /* Solve nonlinear system */
  ierr = FormInitialGuess(&user,x); CHKERRA(ierr);
  ierr = SNESSolve(snes,x,&its);  CHKERRA(ierr); 
  ierr = SNESGetNumberUnsuccessfulSteps(snes,&nfails); CHKERRA(ierr);
  PetscPrintf(PETSC_COMM_SELF,"number of Newton iterations = %d, ",its);
  PetscPrintf(PETSC_COMM_SELF,"number of unsuccessful steps = %d\n\n",nfails);

  /* Free data structures */
  ierr = VecDestroy(x); CHKERRA(ierr); ierr = VecDestroy(f); CHKERRA(ierr);
  ierr = MatDestroy(J); CHKERRA(ierr); ierr = SNESDestroy(snes); CHKERRA(ierr);

  PetscFinalize();
  return 0;
}
/* -------------------------------------------------------------------- */
/*
    FormFunction - Evaluates nonlinear function.
 */

int FormFunction(SNES snes,Vec xvec,Vec fvec,void *ptr)
{
  AppCtx *user = (AppCtx *) ptr;
  int    ierr, nint, n = user->n;
  Scalar *x, *f, param = user->param;

  ierr = VecGetArray(xvec,&x); CHKERRQ(ierr);
  ierr = VecGetArray(fvec,&f); CHKERRQ(ierr);
  switch (user->problem) {
    case 1:
       dsfifj_(&(user->mx),&(user->my),x,f,NULL,&n,"F",&param,1); 
       break;
    case 2:
       dfdcfj_(&(user->mx),&(user->my),x,f,NULL,&n,"F",&param,1);
       break;
    case 3:
       nint = n/8;
       dficfj_(&n,x,f,NULL,&n,"F",&param,&nint,1);
       break;
    case 4:
       nint = n/14;
       dsfdfj_(&n,x,f,NULL,&n,"F",&param,&nint,1);
       break;
    case 5:
       dhhdfj_(&n,x,f,NULL,&n,"F",user->HHVariant,1,5);
       break; 
    case 6:
       dcprfj_(&n,x,f,NULL,&n,"F",1);
       break;
    case 7:
       dcpffj_(&n,x,f,NULL,&n,"F",1);
       break;
    default:
       SETERRQ(1,0,"FormFunction: Invalid problem number");
   }
  ierr = VecRestoreArray(xvec,&x); CHKERRQ(ierr);
  ierr = VecRestoreArray(fvec,&f); CHKERRQ(ierr);
  return 0;
}
/* -------------------------------------------------------------------- */
/*
    FormInitialGuess - Computes initial guess for nonlinear solver.
 */
int FormInitialGuess(AppCtx *user,Vec xvec)
{
  int    ierr, n = user->n, nint;
  Scalar *x, param = user->param;

  ierr = VecGetArray(xvec,&x); CHKERRQ(ierr);
  switch (user->problem) {
    case 1:
       dsfifj_(&(user->mx),&(user->my),x,NULL,NULL,&n,"XS",&param,2);
       break;
    case 2:
       dfdcfj_(&(user->mx),&(user->my),x,NULL,NULL,&n,"XS",&param,2);
       break;
    case 3:
       nint = n/8;
       dficfj_(&n,x,NULL,NULL,&n,"XS",&param,&nint,2);
       break;
    case 4:
       nint = n/14;
       dsfdfj_(&n,x,NULL,NULL,&n,"XS",&param,&nint,2);
       break;
    case 5:
       dhhdfj_(&n,x,NULL,NULL,&n,"XS",user->HHVariant,2,5);
       break; 
    case 6:
       dcprfj_(&n,x,NULL,NULL,&n,"XS",2);
       break;
    case 7:
       dcpffj_(&n,x,NULL,NULL,&n,"XS",2);
       break;
    default:
       SETERRQ(1,0,"FormInitialGuess: Invalid problem number");
   }
  ierr = VecRestoreArray(xvec,&x); CHKERRQ(ierr);
  return 0;
}
/* -------------------------------------------------------------------- */
/*
   FormJacobian - Computes Jacobian matrix with MINPACK-2 routines,
                  using the dense, uniprocessor matrix format.
 */
int FormJacobian(SNES snes,Vec xvec,Mat *J,Mat *PrecJ,MatStructure *flag,
                void *ptr)
{
  AppCtx *user = (AppCtx *) ptr;
  Scalar param = user->param, *Ja, *x;
  int    ierr, nint, n = user->n;

  ierr = VecGetArray(xvec,&x); CHKERRQ(ierr);
  ierr = MatGetArray(*J,&Ja); CHKERRQ(ierr);
  switch (user->problem) {
    case 1:
       dsfifj_(&(user->mx),&(user->my),x,NULL,Ja,&n,"J",&param,1); 
       break;
    case 2:
       dfdcfj_(&(user->mx),&(user->my),x,NULL,Ja,&n,"J",&param,1);
       break;
    case 3:
       nint = n/8;
       dficfj_(&n,x,NULL,Ja,&n,"J",&param,&nint,1);
       break;
    case 4:
       nint = n/14;
       dsfdfj_(&n,x,NULL,Ja,&n,"J",&param,&nint,1);
       break;
    case 5:
       dhhdfj_(&n,x,NULL,Ja,&n,"J",user->HHVariant,1,5);
       break; 
    case 6:
       dcprfj_(&n,x,NULL,Ja,&n,"J",1);
       break;
    case 7:
       dcpffj_(&n,x,NULL,Ja,&n,"J",1);
       break;
    default:
       SETERRQ(1,0,"FormJacobian: Invalid problem number");
  }
  ierr = MatAssemblyBegin(*J,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  ierr = MatAssemblyEnd(*J,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  return 0;
}
/* -------------------------------------------------------------------- */
/*
    UserSetProblem - Initializes test problem.
 */
int UserSetProblem(AppCtx *user,int *argc,char **argv)
{
  int ndim, ierr, flg, problem = 1;

   /* The following are the test problem sizes.  For those that must be a
      multiple of an integer, the multiple is given as a negative value. */
   int Ns[7] = { 0, 0, -8, -14, 8, 5, 11 };

   PetscMemzero(user,sizeof(AppCtx));

  /* Parse and check input arguments */
  ierr = OptionsGetInt(PETSC_NULL,"-p",&problem,&flg); CHKERRQ(ierr);
  if (problem > 7 || problem < 1) SETERRQ(1,0,"UserSetProblem: Invalid problem number");
  switch (problem) {
    case 1: 
      user->param = 6.0; break;   /* lambda */
    case 2: 
      user->param = 100.0; break; /* Reynolds number */
    case 3: 
      user->param = 100.0; break; /* Reynolds number */
    case 4: 
      user->param = 100.0; break; /* viscosity */
   }
  ierr = OptionsGetDouble(PETSC_NULL,"-par",&user->param,&flg); CHKERRQ(ierr);

  /* Set problem dimensions */
  if (Ns[problem-1] > 0) {
    ndim = Ns[problem-1];
    ierr = OptionsGetInt(PETSC_NULL,"-mx",&ndim,&flg); CHKERRQ(ierr);
  } else if (Ns[problem-1] < 0) {
    ndim = 0;
    ierr = OptionsGetInt(PETSC_NULL,"-mx",&ndim,&flg); CHKERRQ(ierr);
    if (!ndim) ndim = -2 * Ns[problem-1];
    else if (ndim % (-Ns[problem-1])) {
      SETERRQ(1,0,"UserSetProblem: Invalid problem size;\n\
       must be a multiple of 8 (problem #3) or 14 (problem #4)");
    }
  } else {
    user->mx = 5; user->my = 4;
    ierr = OptionsGetInt(PETSC_NULL,"-mx",&user->mx,&flg); CHKERRQ(ierr);
    ierr = OptionsGetInt(PETSC_NULL,"-my",&user->my,&flg); CHKERRQ(ierr);
    ndim = user->mx * user->my;
  }
  user->problem = problem;
  user->n       = ndim;
  if (problem == 5) {
    int HHp = 1;
    ierr = OptionsGetInt(PETSC_NULL,"-hp",&HHp,&flg); CHKERRQ(ierr);
    if (HHp < 1 || HHp > 5) SETERRQ(1,0,"UserSetProblem: Invalid human heart problem variant");
    sprintf(user->HHVariant,"DHHD%d",HHp);
    PetscPrintf(PETSC_COMM_SELF,"Problem = %d, variant = %d, N = %d\n",problem,HHp,ndim);
  } else {
    PetscPrintf(PETSC_COMM_SELF,"Problem = %d, param = %g, N = %d\n",problem,user->param,ndim);
  }

  return 0;
}
#else
#include <stdio.h>
int main(int argc,char **args)
{
  fprintf(stdout,"This example does not work for complex numbers.\n");
  return 0;
}
#endif
