#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: ex9.c,v 1.42 1998/03/06 00:16:28 bsmith Exp $";
#endif

static char help[] = "Demonstrates the parallel SLES interface to BlockSolve, and\n\
is adapted from example 1 of BlockSolve.  This problem uses a 3-D grid that\n\
is partitioned in all three dimensions amongst the processors.  The\n\
processors on the left and right ends of the grid use a 7 pt stencil;\n\
those in the middle use a 27-pt stencil. Input arguments are\n\
  -px npx : the number of processors in the x direction\n\
  -py npy : the number of processors in the y direction\n\
  -pz npz : the number of processors in the z direction\n\
  -nx nnx : the number of points in the x direction on 1 processor (nnx>2)\n\
  -ny nny : the number of points in the y direction on 1 processor (nny>2)\n\
  -nz nnz : the number of points in the z direction on 1 processor (nnz>2)\n\
This example requires BlockSolve.\n\n";

#if defined(HAVE_BLOCKSOLVE) && !defined(__cplusplus)

#include "sles.h"
#include "pargrid.h"

int GetMatrix(int,char **,Mat *);
int FillMatrix(Mat,par_grid *,BSprocinfo *);
extern int KSPMonitor_MPIRowbs(KSP,int,double,void *);

int main(int argc,char **args)
{
  Vec     x, u, b;     /* approx solution, exact solution, RHS vector */
  SLES    sles;        /* SLES context */
  PC      pc;          /* PC context */
  Mat     C;           /* matrix */
  int     i, rank, its, grows, gcols, ldim, iglobal, ierr, low, high;
  Scalar  v, none = -1.0, norm;
  MatType mtype;

  PetscInitialize(&argc,&args,(char *)0,help);
  MPI_Comm_rank(PETSC_COMM_WORLD,&rank);

  /* Create and fill matrix */
  ierr = GetMatrix(argc,args,&C); CHKERRA(ierr);

  /* Create vectors and set RHS */
  ierr = MatGetSize(C,&grows,&gcols); CHKERRA(ierr);
  ierr = VecCreateMPI(PETSC_COMM_WORLD,PETSC_DECIDE,grows,&x); CHKERRA(ierr);
  ierr = VecDuplicate(x,&b); CHKERRA(ierr);
  ierr = VecDuplicate(x,&u); CHKERRA(ierr);
  ierr = VecGetLocalSize(x,&ldim); CHKERRA(ierr);
  ierr = VecGetOwnershipRange(x,&low,&high); CHKERRA(ierr);
  for (i=0; i<ldim; i++) {
    iglobal = i + low;
    v = (Scalar)(i + 100*rank);
    ierr = VecSetValues(u,1,&iglobal,&v,INSERT_VALUES); CHKERRA(ierr);
  }
  ierr = VecAssemblyBegin(b); CHKERRA(ierr);
  ierr = VecAssemblyEnd(b); CHKERRA(ierr);
    
  /* Compute right-hand-side */
  ierr = MatMult(C,u,b); CHKERRA(ierr);

  /* Set up linear solver */
  ierr = SLESCreate(PETSC_COMM_WORLD,&sles); CHKERRA(ierr);
  ierr = SLESSetOperators(sles,C,C,0); CHKERRA(ierr);

  /* Set default preconditioner to be ICC (requires rowbs matrix format) */
  ierr = MatGetType(C,&mtype,PETSC_NULL); CHKERRA(ierr);
  if (mtype == MATMPIROWBS) {
    ierr = SLESGetPC(sles,&pc); CHKERRA(ierr);
    ierr = PCSetType(pc,PCICC); CHKERRA(ierr);
  }

  /* Set options from database */
  ierr = SLESSetFromOptions(sles); CHKERRA(ierr);

  if (mtype == MATMPIROWBS) {
    KSP ksp; PCType pcmethod;
    ierr = SLESGetKSP(sles,&ksp); CHKERRA(ierr);
    ierr = PCGetType(pc,&pcmethod); CHKERRA(ierr);
    if (!PetscStrcmp(pcmethod,PCICC)) {
      ierr = KSPSetMonitor(ksp,KSPMonitor_MPIRowbs,(void *)C); CHKERRA(ierr);
    }
  }

  /* Solve linear system */
  ierr = SLESSolve(sles,b,x,&its); CHKERRA(ierr);

  /* Check error */
  ierr = VecAXPY(&none,u,x); CHKERRA(ierr);
  ierr = VecNorm(x,NORM_2,&norm); CHKERRA(ierr);
  PetscPrintf(PETSC_COMM_WORLD,
      "Norm of error %g Number of iterations %d\n",norm,its);  

  /* Free work space */
  ierr = SLESDestroy(sles); CHKERRA(ierr);
  ierr = VecDestroy(x); CHKERRA(ierr);   ierr = VecDestroy(u); CHKERRA(ierr);
  ierr = VecDestroy(b); CHKERRA(ierr);
  ierr = MatDestroy(C); CHKERRA(ierr);
  PetscFinalize();
  return 0;
}
/* ----------------------------------------------------------------- */
/* 
  GetMatrix - Create grid and matrix.
 */
int GetMatrix(int argc,char **args,Mat *newmat)
{
  BSprocinfo *procinfo;
  par_grid   gridA, *grid;
  Scalar     dsend[3], drecv[3];
  Mat        mat;  
  int	     ierr, i, j, k, low, high, count, max_msg_size, size, flg1;
  PetscTruth set;
  point	     *msg;
  MatType    mtype;

  MPI_Comm_size(PETSC_COMM_WORLD,&size);
  grid = &gridA;
  grid->worker_x = 1;
  grid->worker_y = 1;
  grid->worker_z = 1;
  ierr = OptionsGetInt(PETSC_NULL,"-px",&grid->worker_x,&flg1); CHKERRQ(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-py",&grid->worker_y,&flg1); CHKERRQ(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-pz",&grid->worker_z,&flg1); CHKERRQ(ierr);
  if (size != grid->worker_x * grid->worker_y * grid->worker_z) 
    SETERRQ(1,0,"Incompatible number of processors: px * py * pz != size");

  grid->num_x = grid->worker_x * 4;
  grid->num_y = grid->worker_y * 4;
  grid->num_z = grid->worker_z * 4;
  ierr = OptionsGetInt(PETSC_NULL,"-nx",&grid->num_x,&flg1); CHKERRQ(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-ny",&grid->num_y,&flg1); CHKERRQ(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-nz",&grid->num_z,&flg1); CHKERRQ(ierr);

  /* Create BlockSolve communication context */
  procinfo = (BSprocinfo *) BScreate_ctx(); 

  /* local grid size and type */
  /* the end processors are 7-pt stencils, the others are 27-pt */
  if ((Mxpos((grid),(procinfo)) == grid->worker_x-1) ||
                          (Mxpos((grid),(procinfo)) == 0)) {
    grid->l_num_x = grid->num_x;
    grid->type = 7;
  } else {
    grid->l_num_x = grid->num_x / 2;
    grid->type = 27;
  }
  grid->l_num_y = grid->num_y;
  grid->l_num_z = grid->num_z;
  dsend[0] = grid->l_num_x; dsend[1] = grid->l_num_y; dsend[2] = grid->l_num_z;
  MPI_Allreduce(dsend,drecv,3,MPI_DOUBLE,MPI_SUM,PETSC_COMM_WORLD);
  grid->num_x = drecv[0]; grid->num_y = drecv[1]; grid->num_z = drecv[2];
  PetscPrintf(PETSC_COMM_WORLD,"Discretizations (x,y,z): %d %d %d\n",
             grid->num_x,grid->num_y,grid->num_z);

  /* Find the number of local grid points */
  grid->local_total = grid->l_num_x*grid->l_num_y*grid->l_num_z;

  /* Determine the number of global grid points */
  MPI_Allreduce(&(grid->local_total),&(grid->global_total),1,MPI_INT,
                 MPI_SUM,PETSC_COMM_WORLD);

  /* Determine the maximum number of grid points on a single processor */
  MPI_Allreduce(&(grid->local_total),&j,1,MPI_INT, MPI_MAX,PETSC_COMM_WORLD);
  PetscPrintf(PETSC_COMM_WORLD,"Global total = %d, Global Max = %d\n",
			grid->global_total,j);

  /* Create matrix */
  ierr = MatGetTypeFromOptions(PETSC_COMM_WORLD,0,&mtype,&set); CHKERRQ(ierr);
  if (mtype == MATMPIROWBS) {
    ierr = MatCreateMPIRowbs(PETSC_COMM_WORLD,grid->local_total,
           grid->global_total,MAX_LEN,PETSC_NULL,(void *)procinfo,&mat);
    ierr = MatSetOption(mat,MAT_SYMMETRIC);
  } else {
    ierr = MatCreateMPIAIJ(PETSC_COMM_WORLD,grid->local_total,grid->local_total,
           grid->global_total,grid->global_total,MAX_LEN,PETSC_NULL,0,PETSC_NULL,&mat); 
  }
  CHKERRQ(ierr);

  /* This matrix has no i-nodes or cliques */
  BSctx_set_si(procinfo,1); CHKERRQ(0);

  /* Determine the beginning number of my grid points in global numbering */
  ierr = MatGetOwnershipRange(mat,&low,&high); CHKERRQ(ierr);
  grid->offset = low;

  /* Set up grid with ghost points */
  grid->points = (point ***) PetscMalloc(sizeof(point **)*(grid->l_num_x+2));
  for (i=0;i<grid->l_num_x+2;i++) {
    grid->points[i] = (point **) PetscMalloc(sizeof(point *)*(grid->l_num_y+2));
    for (j=0;j<grid->l_num_y+2;j++) {
      grid->points[i][j] = (point *) PetscMalloc(sizeof(point)*(grid->l_num_z+2));
      for (k=0;k<grid->l_num_z+2;k++) {
	grid->points[i][j][k].num = -1;
	grid->points[i][j][k].type = -1;
      }
    }
  }

  /* Number local part of grid */
  count = 0;
  for (i=1;i<grid->l_num_x+1;i++) {
    for (j=1;j<grid->l_num_y+1;j++) {
      for (k=1;k<grid->l_num_z+1;k++) {
	grid->points[i][j][k].num = count + grid->offset;
	grid->points[i][j][k].type = grid->type;
	count++;
      }
    }
  }

  /* Exchange edge information with other processors */
  /* Allocate a message */
  max_msg_size = grid->l_num_x;
  if (max_msg_size < grid->l_num_y) {
    max_msg_size =  grid->l_num_y;
  }
  if (max_msg_size < grid->l_num_z) {
    max_msg_size =  grid->l_num_z;
  }
  max_msg_size += 2;
  max_msg_size *= (max_msg_size*sizeof(point));
  msg = (point *) PetscMalloc(max_msg_size);

  /* now send my east grid edge to the east */
  Msend_border_msg(msg_list,grid->points,msg,EAST_MSG,Meast(grid,procinfo),
		grid->l_num_x,grid->l_num_x,
		0,grid->l_num_y+1,
		0,grid->l_num_z+1,procinfo);

  /* receive from the west */
  Mrecv_border_msg(grid->points,EAST_MSG,
		0,0,
		0,grid->l_num_y+1,
		0,grid->l_num_z+1,procinfo);

  /* now send my west grid edge to the west */
  Msend_border_msg(msg_list,grid->points,msg,WEST_MSG,Mwest(grid,procinfo),
		1,1,
		0,grid->l_num_y+1,
		0,grid->l_num_z+1,procinfo);

  /* receive from the east */
  Mrecv_border_msg(grid->points,WEST_MSG,
		grid->l_num_x+1,grid->l_num_x+1,
		0,grid->l_num_y+1,
		0,grid->l_num_z+1,procinfo);

  /* now send my north grid edge to the north */
  Msend_border_msg(msg_list,grid->points,msg,NORTH_MSG,Mnorth(grid,procinfo),
		0,grid->l_num_x+1,
		grid->l_num_y,grid->l_num_y,
		0,grid->l_num_z+1,procinfo);

  /* receive from the south */
  Mrecv_border_msg(grid->points,NORTH_MSG,
		0,grid->l_num_x+1,
		0,0,
		0,grid->l_num_z+1,procinfo);

  /* now send my south grid edge to the south */
  Msend_border_msg(msg_list,grid->points,msg,SOUTH_MSG,Msouth(grid,procinfo),
		0,grid->l_num_x+1,
		1,1,
		0,grid->l_num_z+1,procinfo);

  /* receive from the north */
  Mrecv_border_msg(grid->points,SOUTH_MSG,
		0,grid->l_num_x+1,
		grid->l_num_y+1,grid->l_num_y+1,
		0,grid->l_num_z+1,procinfo);

  /* now send my upper grid edge up */
  Msend_border_msg(msg_list,grid->points,msg,UP_MSG,Mup(grid,procinfo),
		0,grid->l_num_x+1,
		0,grid->l_num_y+1,
		grid->l_num_z,grid->l_num_z,procinfo);

  /* receive from below */
  Mrecv_border_msg(grid->points,UP_MSG,
		0,grid->l_num_x+1,
		0,grid->l_num_y+1,
		0,0,procinfo);

  /* now send my lower grid edge down */
  Msend_border_msg(msg_list,grid->points,msg,DOWN_MSG,Mdown(grid,procinfo),
		0,grid->l_num_x+1,
		0,grid->l_num_y+1,
		1,1,procinfo);

  /* receive from above */
  Mrecv_border_msg(grid->points,DOWN_MSG,
		0,grid->l_num_x+1,
		0,grid->l_num_y+1,
		grid->l_num_z+1,grid->l_num_z+1,procinfo);

  FINISH_SEND_LIST(msg_list);
  PetscFree(msg);

  /* check to make sure that the entire local grid is numbered */
  for (i=0;i<grid->l_num_x+2;i++) {
    for (j=0;j<grid->l_num_y+2;j++) {
      for (k=0;k<grid->l_num_z+2;k++) {
        if (grid->points[i][j][k].num == -1) {
          PetscPrintf(PETSC_COMM_SELF,"[%d]bad point %d %d %d\n",procinfo->my_id,i,j,k);
        }
      }
    }
  }

  ierr = FillMatrix(mat,grid,procinfo); CHKERRQ(ierr);
  *newmat = mat;
  return 0;
}
/* --------------------------------------------------------------- */
/* 
  FillMatrix - Put values in the matrix
 */
int FillMatrix(Mat mat,par_grid *grid, BSprocinfo *procinfo)
{  
  point	 ***points = grid->points;
  Scalar v;
  int    rstart, rend, grow, ierr;
  int	 i, j, k, ptype, pnum, count;

  MatGetOwnershipRange(mat,&rstart,&rend);
  count = 0;
  for (i=1;i<grid->l_num_x+1;i++) {
    for (j=1;j<grid->l_num_y+1;j++) {
      for (k=1;k<grid->l_num_z+1;k++) {
        /* get a pointer to the correct row */
	ptype = points[i][j][k].type;
        grow = count + rstart;

        /* diagonal */
	pnum = points[i][j][k].num;
        v = 1.0; 
	ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);

	/* east */
	pnum = points[i+1][j][k].num;
	v = -(1.0/8.0);
	ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);

	/* west */
	pnum = points[i-1][j][k].num;
	v = -(1.0/8.0);
	ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);

	/* north */
	pnum = points[i][j+1][k].num;
	v = -(1.0/8.0);
	ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);

	/* south */
	pnum = points[i][j-1][k].num;
	v = -(1.0/8.0);
	ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);

	/* up */
	pnum = points[i][j][k+1].num;
	v = -(1.0/8.0);
	ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);

	/* down */
	pnum = points[i][j][k-1].num;
	v = -(1.0/8.0);
	ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);

	/* east, north, up */
	if ((ptype == 27) || (points[i+1][j+1][k+1].type == 27)) {
	  pnum = points[i+1][j+1][k+1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* east, south, up */
	if ((ptype == 27) || (points[i+1][j-1][k+1].type == 27)) {
	  pnum = points[i+1][j-1][k+1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* east, north, even */
	if ((ptype == 27) || (points[i+1][j+1][k].type == 27)) {
	  pnum = points[i+1][j+1][k].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* east, south, even */
	if ((ptype == 27) || (points[i+1][j-1][k].type == 27)) {
	  pnum = points[i+1][j-1][k].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* east, north, down */
	if ((ptype == 27) || (points[i+1][j+1][k-1].type == 27)) {
	  pnum = points[i+1][j+1][k-1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* east, south, down */
	if ((ptype == 27) || (points[i+1][j-1][k-1].type == 27)) {
	  pnum = points[i+1][j-1][k-1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* east, even, up */
	if ((ptype == 27) || (points[i+1][j][k+1].type == 27)) {
	  pnum = points[i+1][j][k+1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* east, even, down */
	if ((ptype == 27) || (points[i+1][j][k-1].type == 27)) {
	  pnum = points[i+1][j][k-1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* even, north, up */
	if ((ptype == 27) || (points[i][j+1][k+1].type == 27)) {
	  pnum = points[i][j+1][k+1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* even, south, up */
	if ((ptype == 27) || (points[i][j-1][k+1].type == 27)) {
	  pnum = points[i][j-1][k+1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* even, north, down */
	if ((ptype == 27) || (points[i][j+1][k-1].type == 27)) {
	  pnum = points[i][j+1][k-1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* even, south, down */
	if ((ptype == 27) || (points[i][j-1][k-1].type == 27)) {
	  pnum = points[i][j-1][k-1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* west, north, up */
	if ((ptype == 27) || (points[i-1][j+1][k+1].type == 27)) {
	  pnum = points[i-1][j+1][k+1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* west, south, up */
	if ((ptype == 27) || (points[i-1][j-1][k+1].type == 27)) {
	  pnum = points[i-1][j-1][k+1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* west, north, even */
	if ((ptype == 27) || (points[i-1][j+1][k].type == 27)) {
	  pnum = points[i-1][j+1][k].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* west, south, even */
	if ((ptype == 27) || (points[i-1][j-1][k].type == 27)) {
	  pnum = points[i-1][j-1][k].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* west, north, down */
	if ((ptype == 27) || (points[i-1][j+1][k-1].type == 27)) {
	  pnum = points[i-1][j+1][k-1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* west, south, down */
	if ((ptype == 27) || (points[i-1][j-1][k-1].type == 27)) {
	  pnum = points[i-1][j-1][k-1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* west, even, up */
	if ((ptype == 27) || (points[i-1][j][k+1].type == 27)) {
	  pnum = points[i-1][j][k+1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}

	/* west, even, down */
	if ((ptype == 27) || (points[i-1][j][k-1].type == 27)) {
	  pnum = points[i-1][j][k-1].num;
	  v = -(1.0/80.0);
	  ierr = MatSetValues(mat,1,&grow,1,&pnum,&v,INSERT_VALUES); CHKERRQ(ierr);
	}
        count++;
      }
    }
  }
  ierr = MatAssemblyBegin(mat,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  ierr = MatAssemblyEnd(mat,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  return 0;
}
#else
#include "petsc.h"
int main(int argc,char **args)
{
  fprintf(stdout,help);
  return 0;
}
#endif

