#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: gvecsnes.c,v 1.12 1998/04/27 14:20:52 curfman Exp $";
#endif

#include "snes.h"
#include "src/gvec/gvecimpl.h"    /*I "gvec.h" I*/


#undef __FUNC__  
#define __FUNC__ "GVecSolutionSNESMonitor" 
/*@
   GVecSolutionSNESMonitor - Monitors solution at each SNES iteration.

   Collective on SNES

   Input Parameters:
+  snes - the nonlinear solver context
.  it  - the number of iterations so far
.  rnorm - the current (approximate) residual norm
-  ctx - viewer

.seealso: SNESDefaultMonitor(),SNESSetMonitor(),GVecResidualSNESMonitor(),
          GVecErrorSNESMonitor()
@*/
int GVecSolutionSNESMonitor(SNES snes,int it,double rnorm,void* dummy)
{
  Vec    x;
  Viewer ctx = (Viewer)dummy;
  int    ierr;

  PetscFunctionBegin;
  ierr = SNESGetSolution(snes,&x); CHKERRQ(ierr);
  ierr = GVecView(x,ctx); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecResidualSNESMonitor" 
/*@
   GVecResidualSNESMonitor - Monitors residual at each SNES iteration.

   Collective on SNES

   Input Parameters:
+  snes - the nonlinear solver context
.  it  - the number of iterations so far
.  rnorm - the current (approximate) residual norm
-  ctx - viewer

.seealso: SNESDefaultMonitor(),SNESSetMonitor(),GVecSolutionSNESMonitor(),
          GVecErrorSNESMonitor()
@*/
int GVecResidualSNESMonitor(SNES snes,int it,double rnorm,void* dummy)
{
  Vec    x;
  Viewer ctx = (Viewer)dummy;
  int    ierr;

  PetscFunctionBegin;
  ierr = SNESGetFunction(snes,&x); CHKERRQ(ierr);
  ierr = GVecView(x,ctx); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecErrorSNESMonitor" 
/*@
   GVecErrorSNESMonitor - Displays the error at each iteration.

   Collective on SNES

   Input Parameters:
+  snes - the nonlinear context
.  it  - the number of iterations so far
.  rnorm - the current (approximate) residual norm
-  ctx - viewer

   Notes: 
   The final argument to SNESSetMonitor() with this routine must be
   a pointer to a GVecErrorSNESMonitorCtx.

.seealso: SNESDefaultMonitor(),SNESSetMonitor(),,GVecSolutionSNESMonitor(),
          GVecResidualSNESMonitor()
@*/
int GVecErrorSNESMonitor(SNES snes,int it,double rnorm,void* dummy)
{
  GVec                    x,e;
  GVecErrorSNESMonitorCtx *ctx = (GVecErrorSNESMonitorCtx *)dummy;
  int                     ierr;
  Scalar                  mone = -1.0;
  double                  norm_2,norm_max;
  FILE                    *file;
  MPI_Comm                comm;

  PetscFunctionBegin;
  ierr = SNESGetSolution(snes,&x); CHKERRQ(ierr);
  ierr = GVecGetWorkGVec(ctx->solution,&e); CHKERRQ(ierr);
  ierr = VecWAXPY(&mone,x,ctx->solution,e); CHKERRQ(ierr);
  ierr = GVecView(e,ctx->error_viewer); CHKERRQ(ierr);

  /*
      Compute 2-norm and max-norm of error
  */
  if (ctx->norm_error_viewer) {
    ierr = PetscObjectGetComm((PetscObject)snes,&comm); CHKERRQ(ierr);
    ierr = ViewerASCIIGetPointer(ctx->norm_error_viewer,&file); CHKERRQ(ierr);
    ierr = VecNorm(e,NORM_2,&norm_2);CHKERRQ(ierr);
    ierr = VecNorm(e,NORM_MAX,&norm_max);CHKERRQ(ierr);
    PetscFPrintf(comm,file,"Iteration %d residual norm %g error 2-norm %g error max-norm %g\n",
                 it,rnorm,norm_2,norm_max);
  }

  ierr = GVecRestoreWorkGVec(ctx->solution,&e); CHKERRQ(ierr);  
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecSNESOptionsChecker_Private" 
int GVecSNESOptionsChecker_Private(SNES snes) 
{
  char     *prefix;  
  int      ierr,flg,loc[4],nmax;
  MPI_Comm comm;
  Viewer   viewer;
  Draw     draw;

  PetscFunctionBegin;
  ierr = SNESGetOptionsPrefix(snes,&prefix); CHKERRQ(ierr);
  ierr = PetscObjectGetComm((PetscObject)snes,&comm); CHKERRQ(ierr);

  nmax = 4;
  loc[0] = PETSC_DECIDE; loc[1] = PETSC_DECIDE; loc[2] = 300; loc[3] = 300;
  ierr = OptionsGetIntArray(prefix,"-gvec_snes_solutionmonitor",loc,&nmax,&flg); CHKERRQ(ierr);
  if (flg) {
    ierr = ViewerDrawOpenX(comm,0,0,loc[0],loc[1],loc[2],loc[3],&viewer); CHKERRQ(ierr);
    ierr = ViewerDrawGetDraw(viewer,&draw); CHKERRQ(ierr);
    ierr = DrawSetTitle(draw,"Approx. Solution"); CHKERRQ(ierr);
    PLogObjectParent(snes,(PetscObject) viewer);
    ierr = SNESSetMonitor(snes,GVecSolutionSNESMonitor,(void *)viewer); CHKERRQ(ierr);
  }
  nmax = 4;
  loc[0] = PETSC_DECIDE; loc[1] = PETSC_DECIDE; loc[2] = 300; loc[3] = 300;
  ierr = OptionsGetIntArray(prefix,"-gvec_snes_residualmonitor",loc,&nmax,&flg); CHKERRQ(ierr);
  if (flg) {
    ierr = ViewerDrawOpenX(comm,0,0,loc[0],loc[1],loc[2],loc[3],&viewer); CHKERRQ(ierr);
    ierr = ViewerDrawGetDraw(viewer,&draw); CHKERRQ(ierr);
    ierr = DrawSetTitle(draw,"Residual"); CHKERRQ(ierr);
    PLogObjectParent(snes,(PetscObject) viewer);
    ierr = SNESSetMonitor(snes,GVecResidualSNESMonitor,(void *)viewer); CHKERRQ(ierr);
  }
  /*--------------------------------------------------------------------- */
  ierr = OptionsHasName(prefix,"-gvec_snes_fd",&flg); CHKERRQ(ierr);
  if (flg) {
    GMat A,B;
    ierr = SNESGetJacobian(snes,&A,&B,0); CHKERRQ(ierr);
    ierr = SNESSetJacobian(snes,A,B,GVecSNESDefaultComputeJacobianWithColoring,0);CHKERRQ(ierr);
    PLogInfo(snes,"SNESSetFromOptions: Setting gvec finite difference Jacobian matrix\n");
  }
  /*--------------------------------------------------------------------- */
  ierr = OptionsHasName(PETSC_NULL,"-help",&flg); CHKERRQ(ierr);
  if (flg) {
    char pprefix[64];
    PetscStrcpy(pprefix,"-");
    if (prefix)  PetscStrcat(pprefix,prefix);
    (*PetscHelpPrintf)(comm," Additional SNES Monitor options for grid vectors\n",pprefix);
    (*PetscHelpPrintf)(comm,"   %sgvec_snes_solutionmonitor\n",pprefix);
    (*PetscHelpPrintf)(comm,"   %sgvec_snes_residualmonitor\n",pprefix);
  }
  PetscFunctionReturn(0);
}




