
/* $Id: common3and4.c,v 1.9 1998/04/26 21:21:14 bsmith Exp bsmith $ */



/* ------------------------------------------------------------------------------*/
#include <math.h>

/*
     Gnu complex math libraries cannot always compute 0 to some power correctly :-)
*/
#if defined(USE_PETSC_COMPLEX)
static Scalar czero(0.0,0.0);
#define PetscPow(a,b) ( ((a) == czero) ? (czero) : (pow(a,b)) )
#else
#define PetscPow(a,b) pow((a),(b))
#endif

int UserComputeMatrix(GVec xvec,GMat gmat)
{
  Grid                 grid;
  GridData             griddata;
  DA                   da;
  int                  ierr,xs,m,indices[3],M,i,xsg,mg;
  Scalar               values[3],hl,hr,*xvec_x,gv;
  GVec                 xvec_local;
  double               *x;

  /*
     Get the grid data structure
  */
  ierr   = GMatGetGrid(gmat,&grid); CHKERRQ(ierr);
  ierr   = GridGetGridData(grid,&griddata); CHKERRQ(ierr);
  ierr   = GridDataRectangularGetDA(griddata,&da);CHKERRQ(ierr);
  ierr   = GridDataRectangularGetXYZ(griddata,&x,PETSC_NULL,PETSC_NULL);CHKERRQ(ierr);
 
  ierr = GVecGetLocalWorkGVec(xvec,&xvec_local); CHKERRQ(ierr);
  ierr = GVecGlobalToLocal(xvec,INSERT_VALUES,xvec_local); CHKERRQ(ierr);
  ierr = VecGetArray(xvec_local,&xvec_x);CHKERRQ(ierr);

  /*
     Loop over local rows generating stencil 
  */
  ierr = DAGetCorners(da,&xs,0,0,&m,0,0); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(da,&xsg,0,0,&mg,0,0); CHKERRQ(ierr);
  ierr = DAGetInfo(da,0,&M,0,0,0,0,0,0,0,0); CHKERRQ(ierr);
  /*
      Note that values are inserted into the matrix using the 
    global PETSc ordering.
  */
  if (xs == 0) {
    /* special case, left end point of grid */
    i          = 0;
    indices[0] = 0; 
    indices[1] = 1; 
    hr         = x[1] - x[0];
    gv         = PetscPow(xvec_x[0],2.0);
    values[0]  = (-4.0*gv*xvec_x[0]+3*xvec_x[1]*gv)/hr;
    values[1]  = gv*xvec_x[0]/hr;
    ierr = MatSetValues(gmat,1,&i,2,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    xs++; m--;
  }
  if (xs + m == M) {
    /* special case, right end point of grid */
    i          = M-1;
    indices[0] = M-2; 
    indices[1] = M-1; 
    hl         = x[M-1] - x[M-2];
    gv         = PetscPow(xvec_x[mg-1],2.0);
    values[0]  = gv*xvec_x[mg-1]/hl;
    values[1]  = (-4.0*gv*xvec_x[mg-1]+3*xvec_x[mg-2]*gv)/hl;
    ierr = MatSetValues(gmat,1,&i,2,indices,values,INSERT_VALUES);CHKERRQ(ierr);
    m--;
  }
  for ( i=xs; i<xs+m; i++ ) {
    indices[0] = i-1; 
    indices[1] = i; 
    indices[2] = i+1;
    hl         = x[i] - x[i-1];
    hr         = x[i+1] - x[i];
    gv         = PetscPow(xvec_x[i-xsg],2.0);
    values[0]  = gv*xvec_x[i-xsg]/hl;
    values[1]  = -4.*(hr+hl)*gv*xvec_x[i-xsg]/(hr*hl) + 3*gv*xvec_x[i-xsg-1]/hl +
                 3*gv*xvec_x[i-xsg+1]/hr;
    values[2]  = gv*xvec_x[i-xsg]/hr;
    ierr = MatSetValues(gmat,1,&i,3,indices,values,INSERT_VALUES);CHKERRQ(ierr);
  }
  ierr = VecRestoreArray(xvec_local,&xvec_x);CHKERRQ(ierr);
  ierr = GVecRestoreLocalWorkGVec(xvec,&xvec_local); CHKERRQ(ierr);
  ierr = MatAssemblyBegin(gmat,MAT_FINAL_ASSEMBLY);
  ierr = MatAssemblyEnd(gmat,MAT_FINAL_ASSEMBLY);
  return 0;
}

/* ---------------------------------------------------------------------------------*/
/*
     Applies Dirichlet boundary conditions to boundary points of the domain 
*/
int UserComputeDirichlet(GVec inx,GMat A,
                       int (*f)(int,double*,double*,double*,Scalar*,void*),void* fctx)
{
  Grid     grid;
  GridData griddata;
  DA       da;
  int      ierr,istart,iend,rows[2],rowcount = 0,vsize,M;
  Scalar   diagv;
  IS       is;
  double   *x;

  ierr  = GMatGetGrid(A,&grid); CHKERRQ(ierr);
  ierr  = GridGetGridData(grid,&griddata); CHKERRQ(ierr);
  ierr  = GridDataRectangularGetXYZ(griddata,&x,PETSC_NULL,PETSC_NULL);CHKERRQ(ierr);
  ierr  = GridDataRectangularGetDA(griddata,&da); CHKERRQ(ierr);

  ierr = MatGetSize(A,&vsize,&vsize);  CHKERRQ(ierr);
  ierr = MatGetOwnershipRange(A,&istart,&iend); CHKERRQ(ierr);
  ierr = DAGetInfo(da,0,&M,0,0,0,0,0,0,0,0); CHKERRQ(ierr);

  /* 
     Use for diagonal entry the sum of what each endpoint would be with Neumann BC
     this means the diagonal terms on the Dirichlet boundary conditions will be the  
     same order of magnitude as all other diagonal terms
  */
  diagv    = 1.0/(x[M-1] - x[M-2]) + 1.0/(x[1] - x[0]);

  /*  check left end point */
  if (istart == 0) {
    rows[rowcount++] = 0;
  }
  if (iend == vsize) {
    rows[rowcount++] = iend-1;
  }
  ierr = ISCreateGeneral(PETSC_COMM_SELF,rowcount,rows,&is); CHKERRQ(ierr);
  ierr = MatZeroRows(A,is,&diagv); CHKERRQ(ierr);
  ierr = ISDestroy(is); CHKERRQ(ierr);
  return 0;
}

/* --------------------------------------------------------------------------*/

int UserApplyOperator(GVec gvec,GVec result)
{
  Grid      grid;
  GridData  griddata;
  DA        da;
  int       ierr,xs,xsg,m,M,i,mg;
  Scalar    hl,hr,*gvec_x,*result_x;
  GVec      gvec_local,result_local;
  double    *x;
  /*
     Get the grid data structure
  */
  ierr   = GVecGetGrid(gvec,&grid); CHKERRQ(ierr);
  ierr   = GridGetGridData(grid,&griddata); CHKERRQ(ierr);
  ierr   = GridDataRectangularGetXYZ(griddata,&x,PETSC_NULL,PETSC_NULL);CHKERRQ(ierr);
  ierr   = GridDataRectangularGetDA(griddata,&da);CHKERRQ(ierr);

  /*
     Get the local representation of result and gvec
  */
  ierr = GVecGetLocalWorkGVec(result,&result_local); CHKERRQ(ierr);
  ierr = GVecGetLocalWorkGVec(gvec,&gvec_local); CHKERRQ(ierr);
  ierr = GVecGlobalToLocal(gvec,INSERT_VALUES,gvec_local); CHKERRQ(ierr);
  ierr = VecGetArray(gvec_local,&gvec_x);CHKERRQ(ierr);
  ierr = VecGetArray(result_local,&result_x);CHKERRQ(ierr);

  ierr = DAGetCorners(da,&xs,0,0,&m,0,0); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(da,&xsg,0,0,&mg,0,0); CHKERRQ(ierr);
  ierr = DAGetInfo(da,0,&M,0,0,0,0,0,0,0,0); CHKERRQ(ierr);

  /*
     Loop over local rows applying stencil 
  */
  if (xs + m == M) {
    /* special case, right end point of grid */
    result_x[mg-1] = PetscPow(gvec_x[mg-1],3.0)*(-gvec_x[mg-1] + gvec_x[mg-2])/(x[M-1]-x[M-2]);
    m--;
  }
  if (xs == 0) {
    /* special case, left end point of grid */
    result_x[0] = PetscPow(gvec_x[0],3.0)*(-gvec_x[0] + gvec_x[1])/(x[1]-x[0]);
    xs++; m--;
  }
  for ( i=xs; i<xs+m; i++ ) {
    hl              = x[i] - x[i-1];
    hr              = x[i+1] - x[i];
    result_x[i-xsg] = PetscPow(gvec_x[i-xsg],3.0)*(gvec_x[i-1-xsg]/hl - 
                                             ((hr+hl)/(hr*hl))*gvec_x[i-xsg] +
                                              gvec_x[i+1-xsg]/hr) + .125*(hr+hl);
  }

  ierr = VecRestoreArray(gvec_local,&gvec_x);CHKERRQ(ierr);
  ierr = VecRestoreArray(result_local,&result_x);CHKERRQ(ierr);
  ierr = GVecLocalToGlobal(result_local,INSERT_VALUES,result); CHKERRQ(ierr);
  ierr = GVecRestoreLocalWorkGVec(result,&result_local); CHKERRQ(ierr);
  ierr = GVecRestoreLocalWorkGVec(gvec,&gvec_local); CHKERRQ(ierr);
  return 0;
}

/*----------------------------------------------------------------------------*/
int UserApplyDirichlet(GVec x,GVec F,
                       int (*f)(int,double*,double*,double*,Scalar*,void*),void* fctx)
{
  Grid      grid;
  GridData  griddata;
  DA        da;
  int       ierr,istart,iend,vsize,m,M;
  Scalar    value,*xx,diagv;
  double    *xgrid;

  ierr = GVecGetGrid(x,&grid); CHKERRQ(ierr);
  ierr = GridGetGridData(grid,&griddata);CHKERRQ(ierr);
  ierr = GridDataRectangularGetXYZ(griddata,&xgrid,PETSC_NULL,PETSC_NULL);CHKERRQ(ierr);
  ierr = GridDataRectangularGetDA(griddata,&da);CHKERRQ(ierr);
  ierr = VecGetSize(x,&vsize);  CHKERRQ(ierr);
  ierr = VecGetOwnershipRange(x,&istart,&iend); CHKERRQ(ierr);
  ierr = DAGetInfo(da,0,&M,0,0,0,0,0,0,0,0); CHKERRQ(ierr);

  /* 
     Use for diagonal entry the sum of what each endpoint would be with Neumann BC
     this means the diagonal terms on the Dirichlet boundary conditions will be the  
     same order of magnitude as all other diagonal terms
  */
  diagv    = 1.0/(xgrid[M-1] - xgrid[M-2]) + 1.0/(xgrid[1] - xgrid[0]);

  if (istart == 0) {
    /*
       The nonlinear function on the Dirichlet points is 
          C*(x - u_Dirichlet)
    */
    ierr    = (*f)(1,xgrid,0,0,&value,fctx); CHKERRQ(ierr);
    ierr    = VecGetArray(x,&xx); CHKERRQ(ierr);
    value   = diagv*(xx[0] - value);
    ierr    = VecSetValues(F,1,&istart,&value,INSERT_VALUES); CHKERRQ(ierr);
  }
  if (iend == vsize) {
    iend--;
    ierr    = (*f)(1,xgrid+iend,0,0,&value,fctx); CHKERRQ(ierr);
    ierr    = VecGetArray(x,&xx); CHKERRQ(ierr);
    ierr    = VecGetLocalSize(x,&m); CHKERRQ(ierr);
    value   = diagv*(xx[m-1] - value);
    ierr    = VecSetValues(F,1,&iend,&value,INSERT_VALUES); CHKERRQ(ierr);
  }
  return 0;
}





