#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: dfvec.c,v 1.80 1998/04/27 16:06:47 curfman Exp $";
#endif
/*
  This file provides some utility routines for manipulating discrete functions
  (vectors that are associated with grids, possibly with multiple degrees of
  freedom per node). 

  This component is new; changes in the user interface will be occuring in the
  near future!
*/

#include "src/dfvec/dfvimpl.h"    /*I "dfvec.h" I*/

extern int DFSetSeqRegularOps_Private(DF);
extern int DFSetSeqGeneralOps_Private(DF);
extern int DFSetMPIGeneralOps_Private(DF);

#undef __FUNC__  
#define __FUNC__ "DFVecShellAssociate" 
/*@
   DFVecShellAssociate - Associates a DF shell with a vector.

   Input Parameters:
+  df - DF shell, formed by DFShellCreate()
-  vec - vector

   Notes:
   Upon return, the vector is effectively transformed to a DFVec object,
   so that it can be used for any DFVecXXX() operation as well as any
   VecXXX() operation.
   
.keywords: discrete function, shell, vector, associate

.seealso: DFShellCreate()
@*/
int DFVecShellAssociate(DF df,Vec vec)
{
  int  ierr, flag;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(df,DF_COOKIE);
  PetscValidHeaderSpecific(vec,VEC_COOKIE);
  ierr = MPI_Comm_compare(vec->comm,df->comm,&flag);CHKERRQ(ierr);
  if (flag == MPI_UNEQUAL) {
    SETERRQ(PETSC_ERR_ARG_INCOMP,0,"Incompatible communicators for Vec and DF contexts");
  }
  /* We may want to change this to copy the DF shell as well */
  ierr = PetscObjectCompose((PetscObject)vec,"df",(PetscObject)df);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecGetDFShell" 
/*@
   DFVecGetDFShell - Gets a DF shell from a vector.

   Input Parameters:
+  v - the vector
-  df - the associated DF shell

.keywords: discrete function, vector, get, shell
@*/
int DFVecGetDFShell(DFVec v,DF *df)
{
  int  ierr;
  void *dfv;
  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  ierr = PetscObjectQuery((PetscObject)v,"df",(PetscObject*)&dfv); CHKERRQ(ierr);
  if (dfv == PETSC_NULL) SETERRQ(PETSC_ERR_ARG_WRONG,0,"This is not a DFVec object! Null DF shell!");
  *df = (DF)dfv;
  PetscValidHeaderSpecific(*df,DF_COOKIE);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecCopy"
/*@
   DFVecCopy - Copies a DF vector. 

   Input Parameter:
.  x - the DF vector

   Output Parameter:
.  y - the DF vector copy

   Notes:
   DFVecCopy() should be used for copying DFVecs instead of the usual VecCopy()
   for standard vectors() if one wants to copy the extra DF data.

.keywords: discrete function, vector, copy
@*/
int DFVecCopy(DFVec x,DFVec y)
{
  int ierr;
  DF  xdf, ydf;
  PetscFunctionBegin;
  ierr = VecCopy(x,y); CHKERRQ(ierr);
  ierr = DFVecGetDFShell(x,&xdf); CHKERRQ(ierr);
  ierr = DFVecGetDFShell(y,&ydf); CHKERRQ(ierr);
  if (xdf->ops->copy) {ierr = (*xdf->ops->copy)(xdf,ydf); CHKERRQ(ierr);}
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFRefineCoordinates"
/*
    DFRefineCoordinates - Creates a new set of grid points that are refinements
    of those that have been set with DFShellSetCoordinates().

    Input Parameters:
.   df - the initial DF shell
.   dfref - the refined DF shell

    Notes:
    Called by DFVecRefineVector().

    Currently, we fully support only 3D structured meshes with
    brick elements (2D rectangular meshes), where c0, c1, and
    c2 correspond on each processor to the x-, y-, and z-coordinates
    for the global mesh.

    At present, the only refinement we support is doubling the number
    of grid spacings in each direction.

.seealso: DFVecSetCoordinates(), DFVecRefineVector()
*/
int DFRefineCoordinates(DF df,DF dfref)
{
  int    ierr;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(df,DF_COOKIE);
  PetscValidHeaderSpecific(dfref,DF_COOKIE);
  if (!df->ops->refinecoordinates) SETERRQ(PETSC_ERR_SUP,0,"");
  ierr = (*df->ops->refinecoordinates)(df,dfref); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}
#undef __FUNC__  
#define __FUNC__ "DFVecRefineVector"
/*@
    DFVecRefineVector - Creates a new vector that is a refinement of
    a given vector.

    Input Parameters:
.   v - discrete function vector

    Output Parameters:
.   vref - refined discrete function vector

    Note:
    Currently, we fully support refinement only for vectors that are
    defined on 3D structured meshes with brick elements (2D rectangular
    meshes), such as are created by the PETSc distributed arrays (DAs)
    for the parallel case.

    At present, the only refinement we support is doubling the number
    of grid spacings in each direction.
@*/
int DFVecRefineVector(DFVec v,DFVec *vref)
{
  int ierr;
  DF  df, dfref;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  ierr = DFVecGetDFShell(v,&df); CHKERRQ(ierr);
  if (!df->ops->refinevector) SETERRQ(PETSC_ERR_SUP,0,"");
  PLogEventBegin(DFVec_RefineVector,v,0,0,0);
  ierr = (*df->ops->refinevector)(v,vref); CHKERRQ(ierr);
  PLogEventEnd(DFVec_RefineVector,v,0,0,0);

  if (df->cset) { /* Refine coordinates of vector if they've been set */
    ierr = DFVecGetDFShell(*vref,&dfref); CHKERRQ(ierr);
    ierr = DFRefineCoordinates(df,dfref); CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecView_Private" 
int DFVecView_Private(DFVec dfv,Viewer viewer)
{
  Vec        v = (Vec) dfv;
  DF         df;
  ViewerType vtype;
  int        ierr;

  PetscFunctionBegin;
  if (!viewer) { /* so that viewers may be used from debuggers */
    viewer = VIEWER_STDOUT_SELF;
  }
  ierr = ViewerGetType(viewer,&vtype); CHKERRQ(ierr);
  ierr = DFVecGetDFShell(dfv,&df); CHKERRQ(ierr);
  ierr = VecView(v,viewer); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecAssembleFullVector"
/*@
    DFVecAssembleFullVector - Assembles full multicomponent vector as a
    combination of subvectors.

    Input Parameters:
.   vsub - component vectors

    Output Parameter:
.   v - full discrete function vector

.seealso: DFVecGetComponentVectors()
@*/
int DFVecAssembleFullVector(Vec *vsub,DFVec v)
{
  int ierr, i;
  DF  df;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  ierr = DFVecGetDFShell(v,&df); CHKERRQ(ierr);
  for (i=0; i<df->nc; i++) {PetscValidHeaderSpecific(vsub[i],VEC_COOKIE);}
  if (!df->ops->assemblefullvector) SETERRQ(PETSC_ERR_SUP,0,"");

  PLogEventBegin(DFVec_AssembleFullVector,vsub,v,0,0);
  ierr = (*df->ops->assemblefullvector)(vsub,v); CHKERRQ(ierr);
  PLogEventEnd(DFVec_AssembleFullVector,vsub,v,0,0);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFShellCreate" 
/*@C
   DFShellCreate - Creates a discrete function object.

   Input Parameters:
+  comm - MPI communicator
.  type - type of discrete function, 
        one of DF_SEQGEN (general sequential), DF_SEQREG (regular
        sequential), or DF_MPIGEN (general parallel)
.  dim - number of problem dimensions (1, 2, or 3)
.  nc - number of components per grid point
.  ord - grid component ordering, either ORDER_1 (ordering by components
         most rapidly) or ORDER_2 (ordering 1 component for whole grid, 
         then the next component, etc.)
.  label - optional character string associated with each component
-  n0, n1, n2 - number of grid points in each dimension

   Output Parameter:
.  df - discrete function (DF) shell

   Notes:  
   When using PETSc distributed arrays (DAs), the vectors obtained from
   DACreateGlobalVector() and DACreateLocalVector() are automatically
   discrete function vectors (DAVec's) of the type DF_MPIREG.  Thus,
   users of DAs need not be concerned with DFShellCreate() at all,
   since this routine is intended for use with more general types of
   vectors.

.seealso: DFShellDestroy(), DFShellSetCoordinates()
@*/
int DFShellCreate(MPI_Comm comm,DFType type,int dim,int nc,DFComponentOrdering ord,
                 char **label,int n0,int n1,int n2,DF* df)
{
  DF   v;
  int  i, ierr, flag;
  char *vl;

  PetscFunctionBegin;

  if (dim != 1 && dim != 2 && dim != 3) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"only dim=1,2,3 supported");
  if (ord != ORDER_1 && ord != ORDER_2) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"invalid ordering");
  if (n0 <= 0 || n1 <= 0 || n2 <= 0 || nc <= 0) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"nc, n0, n1, n2 > 0 only");
  if (type == DF_SEQGEN || type == DF_SEQREG) {
    ierr = MPI_Comm_compare(PETSC_COMM_SELF,comm,&flag);CHKERRQ(ierr);
    if (flag == MPI_UNEQUAL) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Must call with MPI_COMM_SELF or PETSC_COMM_SELF");
  } else if (type == DF_MPIGEN) {
    ; /* do nothing */
  } else if (type == DF_MPIREG) { SETERRQ(PETSC_ERR_ARG_WRONG,0,
   "DF_MPIREG is valid only with distributed arrays. Discrete function\n\
    is automatically formed with DACreate[1,2,3]d(). Extract DF with DFVecGetDFShell().");
  } else SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Unsupported DFType");

  if (type == DF_SEQGEN) {
    PetscHeaderCreate(v,_p_DF,struct _DFOps,DF_COOKIE,DF_SEQGEN,comm,DFShellDestroy,0);
  } else if (type == DF_SEQREG) {
    PetscHeaderCreate(v,_p_DF,struct _DFOps,DF_COOKIE,DF_SEQREG,comm,DFShellDestroy,0);
  } else if (type == DF_MPIGEN) {
    PetscHeaderCreate(v,_p_DF,struct _DFOps,DF_COOKIE,DF_MPIGEN,comm,DFShellDestroy,0);
  } else SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Unsppported DFType");
  PLogObjectCreate(v);

  PLogObjectMemory(v,sizeof(struct _p_DF));
  v->dim     = dim;
  v->da_user = 0;
  v->da      = 0;
  v->nc      = nc;
  v->order   = ord;
  v->cflag   = 0;
  if (label != PETSC_NULL) {
    v->label = label; v->ldefault = 0;
  } else {
    v->label = (char **) PetscMalloc(nc*sizeof(char*)); CHKPTRQ(v->label);
    for (i=0; i<nc; i++) {
      v->label[i] = vl = (char *) PetscMalloc(25*sizeof(char)); CHKPTRQ(vl);
      sprintf(vl,"Vector Component %d",i);
    }
    v->ldefault = 1;
    PLogObjectMemory(v,(nc*(sizeof(char*)+25*sizeof(char))));
  }
  v->coord[0]  = 0;  v->coord[1] = 0; v->coord[2] = 0;
  v->cset      = 0;
  v->rset      = 0;
  v->calloc    = 0;
  v->user      = 0;
  v->user_copy = 0;
  v->df_local  = 0;
  v->lnp[0]    = n0;        v->lnp[1]  = n1;       v->lnp[2]   = n2;
  v->gnp[0]    = v->lnp[0]; v->gnp[1] = v->lnp[1]; v->gnp[2] = v->lnp[2];

  if (type == DF_SEQGEN) {
    ierr = DFSetSeqGeneralOps_Private(v); CHKERRQ(ierr);
  } else if (type == DF_SEQREG) {
    ierr = DFSetSeqRegularOps_Private(v); CHKERRQ(ierr);
  } else if (type == DF_MPIGEN) {
    ierr = DFSetMPIGeneralOps_Private(v); CHKERRQ(ierr);
  } else SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Unsupported DFType");
  v->ops->view    = DFVecView_Private;
  *df = v;
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFShellGetLocalDFShell" 
/*@
   DFShellGetLocalDFShell - Gets the local DF shell from a parallel DF shell.

   Input Parameter:
.  df - the global DF shell

   Output Parameter:
.  df_local - the associated local DF shell

.keywords: discrete function, get, local, shell
@*/
int DFShellGetLocalDFShell(DF df,DF *df_local)
{
  DFType type;
  PetscFunctionBegin;
  PetscValidHeaderSpecific(df,DF_COOKIE);
  type = (DFType)df->type;
  if (type == DF_MPIREG) *df_local = (DF)df->df_local;
  else *df_local = df;
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFShellGetInfo" 
/*@ 
   DFShellGetInfo - Gets info about a DF shell.

   Input Parameter:
.  df - DF shell

   Output Parameters:
+  type - type of discrete function, currently one
         of DF_SEQGEN, DF_SEQREG, DF_MPIGEN, or DF_MPIREG
.  dim - number of problem dimensions (1, 2, or 3)
.  nc - number of components per grid point
.  ord - grid component ordering, either ORDER_1 (ordering by components 
         most rapidly), or ORDER_2 (ordering 1 component for whole grid, 
         then the next component, etc.)
-  n0, n1, n2 - number of local grid points in each dimension

.seealso: DFVecView()
@*/
int DFShellGetInfo(DF df,DFType *type,int *dim,int *nc,DFComponentOrdering *ord,
               int *n0,int *n1,int *n2)
{
  PetscFunctionBegin;
  if (type) *type = (DFType)df->type;
  if (dim)  *dim  = df->dim;
  if (nc)   *nc   = df->nc;
  if (ord)  *ord  = df->order;
  if (n0)   *n0   = df->lnp[0];
  if (n1)   *n1   = df->lnp[1];
  if (n2)   *n2   = df->lnp[2];
  PetscFunctionReturn(0);
}

extern int DFShellCreateDA_Private(MPI_Comm,char**,DA,DF*);
#undef __FUNC__  
#define __FUNC__ "DFShellDuplicate"
/*@
   DFShellDuplicate - Duplicates a DF shell.

   Input Parameter:
.  df1 - original DF shell

   Output Parameter:
.  df2 - new DF shell

.seealso: DFShellCreate()
@*/
int DFShellDuplicate(DF df1,DF* df2)
{
  int    ierr, i;
  DF     vg2;
  DFType type;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(df1,DF_COOKIE);  

  type = (DFType)df1->type;
  if (type == DF_MPIREG) {
    ierr = DFShellCreateDA_Private(df1->comm,df1->label,df1->da_user,&vg2); CHKERRQ(ierr);
  }
  else if (type == DF_SEQGEN || type == DF_SEQREG || type == DF_MPIGEN) {
    ierr = DFShellCreate(df1->comm,type,df1->dim,df1->nc,df1->order,
           df1->label,df1->lnp[0],df1->lnp[1],df1->lnp[2],&vg2); CHKERRQ(ierr);
  } else SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Unknown DFType");

  *df2          = vg2;
  vg2->ldefault = 1;
  vg2->gnp[0] = df1->gnp[0]; vg2->gnp[1] = df1->gnp[1]; vg2->gnp[2] = df1->gnp[2];
  vg2->label = (char **) PetscMalloc(vg2->nc*sizeof(char*)); CHKPTRQ(vg2->label);
  for (i=0; i<vg2->nc; i++) {
    vg2->label[i] = (char *) PetscMalloc(25*sizeof(char*)); CHKPTRQ(vg2->label[i]);
    if (!df1->cflag) PetscStrcpy(vg2->label[i],df1->label[i]);
  }
  vg2->user_copy = df1->user_copy;
  if (df1->user && df1->user_copy) (*vg2->user_copy)(df1->user,&vg2->user);
  
  PetscFunctionReturn(0);
}
#undef __FUNC__  
#define __FUNC__ "DFShellSetCoordinates"
/*@
    DFShellSetCoordinates - Sets grid coordinate information in a DF shell.

    Input Parameters:
+   df - the DF shell
.   c0, c1, c2 - coordinates in each dimension
-   M, N, P - number of coordinates in each dimension

    Notes:
    The coordinate information is used only in the routines
    DFVecRefineVector(), DFVecDrawTensorContoursX(), and
    DFVecDrawTensorSurfaceContoursVRML().

    Currently, we fully support only 3D structured meshes with
    brick elements (2D rectangular meshes), where the user
    sets on each processor the x-, y-, and z-coordinates (in
    c0, c1, and c2, respectively) for the entire global mesh.

    Example:
    3-D grid of global dimension MxNxP:
.vb
      for ( i=0; i<M; i++ ) c0[i] = i*0.3;
      for ( i=0; i<N; i++ ) c1[i] = i*0.6;
      for ( i=0; i<P; i++ ) c2[i] = i*0.5;
.ve

.seealso: DFShellCreate(), DFShellGetCoordinates()
@*/
int DFShellSetCoordinates(DF df,int M,int N,int P,double *c0,double *c1,double *c2)
{
  DFType type;
  int    ierr;
  PetscFunctionBegin;
  PetscValidHeaderSpecific(df,DF_COOKIE);
  ierr = PetscObjectGetType((PetscObject)df,(int*)&type); CHKERRQ(ierr);
  if (type == DF_SEQREG || type == DF_MPIREG) {
    if (M != df->gnp[0]) SETERRQ(PETSC_ERR_ARG_WRONG,0,"M != value set in DFShellCreate()");
    if (df->dim >= 2 && N != df->gnp[1])
      SETERRQ(PETSC_ERR_ARG_WRONG,0,"N != value set in DFShellCreate()");
    if (df->dim >= 3 && P != df->gnp[2]) 
      SETERRQ(PETSC_ERR_ARG_WRONG,0,"P != value set in DFShellCreate()");
  }
  df->coord[0] = c0;
  df->coord[1] = c1;
  df->coord[2] = c2;
  df->cset = 1;
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFShellGetCoordinates"
/*@
    DFShellGetCoordinates - Gets grid coordinate information.

    Input Parameters:
+   df - the DF shell
.   c0, c1, c2 - coordinates in each dimension
-   M, N, P - number of coordinates in each dimension

    Note:
    Currently, we fully support only 3D structured meshes with
    brick elements (2d rectangular meshes), where c0, c1, and
    c2 correspond on each processor to the x-, y-, and z-coordinates
    for the global mesh.

.seealso: DFShellSetCoordinates()
@*/
int DFShellGetCoordinates(DF df,int *M,int *N,int *P,double **c0,double **c1,double **c2)
{
  DFType type;
  PetscFunctionBegin;
  PetscValidHeaderSpecific(df,DF_COOKIE);
  *c0 = df->coord[0];
  *c1 = df->coord[1];
  *c2 = df->coord[2];
  PetscObjectGetType((PetscObject)df,(int*)&type);
  if (type == DF_SEQREG || type == DF_MPIREG) {
    *M = df->gnp[0]; *N = df->gnp[1]; *P = df->gnp[2];
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFShellDestroy" 
/*@
    DFShellDestroy - Destroys DF shell.

    Input Parameters:
.   df - a DF shell

.seealso: DFShellCreate()
@*/
int DFShellDestroy(DF df)
{
  int i, ierr;
  PetscFunctionBegin;
  PetscValidHeaderSpecific(df,DF_COOKIE);

  if (--df->refct > 0) PetscFunctionReturn(0); 

  if (df->ldefault) {
    for (i=0; i<df->nc; i++)
      if (df->label[i]) PetscFree(df->label[i]);
    PetscFree(df->label);
  }
  if (df->calloc) {
    if (df->coord[0]) PetscFree(df->coord[0]);
    if (df->coord[1]) PetscFree(df->coord[1]);
    if (df->coord[2]) PetscFree(df->coord[2]);
  }
  if (df->df_local) {ierr = DFShellDestroy((DF)df->df_local); CHKERRQ(ierr);}
  PLogObjectDestroy(df);
  PetscHeaderDestroy(df);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecDrawTensorContoursX" 
/*@
   DFVecDrawTensorContoursX - Draws contour plot of each component within a
   multi-component vector.
  
   Input Parameters:
+  v - the DF vector
-  width, height - the width and height in pixels of each
                   components screen (same for each)

   Options Database Keys:
+  -nox - Disables all x-windows output
.  -display <name> - Indicates name of machine for the X display
-  -draw_pause <sec> - Sets number of seconds to pause after each component
         has been drawn; use -1 to indicate pausing until user input.

   Notes:
   Currently this routine is supports 2D structured problems only, where
   DrawTensorContour() is called for each component of the vector.

   The user is responsible for making sure that the input vector is
   completely updated and ready for viewing.

   If DFShellSetCoordinates() has not been called to set the grid 
   coordinates, a uniform grid over [0,1] X [0,1] is used for the
   contour plot.

   This routine is currently disabled for the complex numbers version.

.seealso: DrawTensorContour()
@*/
int DFVecDrawTensorContoursX(DFVec v,int width,int height)
{
  int  ierr;
  DF   df;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  ierr = DFVecGetDFShell(v,&df); CHKERRQ(ierr);
  if (!df->ops->drawcontours) SETERRQ(PETSC_ERR_SUP,0,"");
  if (df->dim != 2) {
    (*PetscErrorPrintf)("DFVecDrawTensorContoursX: Supports 2D grids only\n");
    PetscFunctionReturn(0);
  }
  PLogEventBegin(DFVec_DrawContours,v,0,0,0);
  ierr = (*df->ops->drawcontours)(v,width,height); CHKERRQ(ierr);
  PLogEventEnd(DFVec_DrawContours,v,0,0,0);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecDrawTensorContours" 
/*@
   DFVecDrawTensorContours - Draws contour plot of each component within a
   multi-component vector.
  
   Input Parameters:
+  v - the DF vector
-  viewer - the viewer, created with ViewerDrawOpenX(), to display in


   Options Database Keys:
+  -nox - Disables all x-windows output
.  -display <name> - Indicates name of machine for the X display
-  -draw_pause <sec> - Sets number of seconds to pause after each component
         has been drawn; use -1 to indicate pausing until user input.

   Notes:
   Currently this routine is supports 2D structured problems only, where
   DrawTensorContour() is called for each component of the vector.

   The user is responsible for making sure that the input vector is
   completely updated and ready for viewing.

   If DFShellSetCoordinates() has not been called to set the grid 
   coordinates, a uniform grid over [0,1] X [0,1] is used for the
   contour plot.

   This routine is currently disabled for the complex numbers version.

.seealso: DrawTensorContour()
@*/
int DFVecDrawTensorContours(DFVec v,Viewer *viewer)
{
  int  ierr;
  DF   df;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  ierr = DFVecGetDFShell(v,&df); CHKERRQ(ierr);
  if (!df->ops->drawcontoursviewer) SETERRQ(PETSC_ERR_SUP,0,"");
  if (df->dim != 2) {
    (*PetscErrorPrintf)("DFVecDrawTensorContours: Supports 2D grids only\n");
    PetscFunctionReturn(0);
  }
  PLogEventBegin(DFVec_DrawContours,v,0,0,0);
  ierr = (*df->ops->drawcontoursviewer)(v,viewer); CHKERRQ(ierr);
  PLogEventEnd(DFVec_DrawContours,v,0,0,0);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecDrawTensorSurfaceContoursVRML" 
/*@
   DFVecDrawTensorSurfaceContoursVRML - Draws a VRML surface contour plot of each
   component within a multi-component vector.
  
   Input Parameters:
.  v - the DF vector

   Options Database Keys:
+  -dfvec_vrml_nocontour - draws surface only (no contour)
-  -dfvec_vrml_basefile <filename> - base filename for VRML output
       (default is "dfvec"); output file names are <filename>.<num>.vrml, 
       where <num> is the component number for vectors with multiple degrees
       of freedom per node.

   Notes:
   Currently this routine supports 2D and 3D structured problems only, where
   DrawTensorSurfaceContour_VRML() is called for each component of the vector.
   For 3D problems, the data is viewed in layers in the z-direction. We
   eventually should generalize this.

   If DFShellSetCoordinates() has not been called to set the grid 
   coordinates, a uniform grid over [0,1] X [0,1] is used for the
   contour plot.

   This routine is currently disabled for the complex numbers version.
@*/
int DFVecDrawTensorSurfaceContoursVRML(DFVec v)
{
  int  ierr;
  DF   df;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  ierr = DFVecGetDFShell(v,&df); CHKERRQ(ierr);
  if (!df->ops->drawcontoursvrml) SETERRQ(PETSC_ERR_SUP,0,"");
  if (df->dim != 2 && df->dim != 3) {
    (*PetscErrorPrintf)("DFVecDrawTensorSurfaceContoursVRML: Supports 2D and 3D grids only\n");
    PetscFunctionReturn(0);
  }
  PLogEventBegin(DFVec_DrawContours,v,0,0,0);
  ierr = (*df->ops->drawcontoursvrml)(v); CHKERRQ(ierr);
  PLogEventEnd(DFVec_DrawContours,v,0,0,0);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecGetComponentVectors" 
/*@C
   DFVecGetComponentVectors - Extracts the component vectors from a larger
   multicomponent vector.  We currently assume that all components vectors are
   of equal length.

   Input Parameter:
.  v - the DF vector

   Output Parameters:
+  ncomp - number of component vectors
-  vcomp - array of component vectors

.seealso: DFVecCreate()
C@*/
int DFVecGetComponentVectors(DFVec v,int *ncomp,DFVec **vcomp)
{
  int        ierr, i, gsize, lsize, nc;
  Vec        v1, *vsub;
  VecScatter vsctx;
  VecType    type;
  IS         newis, is;
  DF         df;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(v,VEC_COOKIE);
  ierr = DFVecGetDFShell(v,&df); CHKERRQ(ierr);
  PLogEventBegin(DFVec_GetComponentVectors,v,0,0,0);
  *ncomp = nc = df->nc;
  ierr = VecGetSize(v,&gsize); CHKERRQ(ierr);
  ierr = VecGetLocalSize(v,&lsize); CHKERRQ(ierr);
  if (gsize%nc || lsize%nc)  /* assume same length for each subvector */
    SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"invalid number of components");
  gsize /= nc; lsize /= nc;
  ierr = VecGetType(v,&type,PETSC_NULL); CHKERRQ(ierr);
  if (type == VECSEQ) {
    ierr = VecCreateSeq(PETSC_COMM_SELF,gsize,&v1); CHKERRQ(ierr);
  }
  else if (type == VECMPI) {
    ierr = VecCreateMPI(df->comm,lsize,gsize,&v1); CHKERRQ(ierr);
  }
  else SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"vector type not supported");
  PLogObjectParent(df,v1);
  ierr = VecDuplicateVecs(v1,df->nc,&vsub); CHKERRQ(ierr);
  PLogObjectParents(df,df->nc,vsub);
  ierr = VecDestroy(v1); CHKERRQ(ierr);
  if (nc == 1) {
    ierr = VecCopy(v,vsub[0]); CHKERRQ(ierr);
  }
  else {
    if (df->order != ORDER_1 && df->order != ORDER_2) 
      SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Ordering not supported.");
    ierr = ISCreateStride(df->comm,gsize,0,1,&newis); CHKERRQ(ierr);
    PLogObjectParent(df,newis);
    for (i=0; i<nc; i++) {
      if (df->order == ORDER_1) {
        ierr = ISCreateStride(df->comm,gsize,i,nc,&is); CHKERRQ(ierr);
      }
      else if (df->order == ORDER_2) {
        ierr = ISCreateStride(df->comm,gsize,i*gsize,1,&is); CHKERRQ(ierr);
      }
      PLogObjectParent(df,is);
      ierr = VecScatterCreate(v,is,vsub[i],newis,&vsctx); CHKERRQ(ierr);
      PLogObjectParent(df,vsctx);
      ierr = VecScatterBegin(v,vsub[i],INSERT_VALUES,SCATTER_FORWARD,vsctx);CHKERRQ(ierr);
      ierr = VecScatterEnd(v,vsub[i],INSERT_VALUES,SCATTER_FORWARD,vsctx); CHKERRQ(ierr);
      ierr = VecScatterDestroy(vsctx); CHKERRQ(ierr);
      ierr = ISDestroy(is); CHKERRQ(ierr);
    }
    ierr = ISDestroy(newis); CHKERRQ(ierr);
  }
  *vcomp = vsub;
  PLogEventEnd(DFVec_GetComponentVectors,v,0,0,0);
  PetscFunctionReturn(0);
}

/* #include "pinclude/pviewer.h" */
#undef __FUNC__  
#define __FUNC__ "DFVecView" 
/*@ 
   DFVecView - Views a discrete function vector.

   Input Parameters:
.  v - the DF vector
.  viewer - an optional visualization context

   Notes:
   DFVecView() supports the same viewers as VecView().  The only difference
   is that for all multiprocessor cases, the output vector employs the same
   ordering that would have been used for the uniprocessor case.

   The available visualization contexts include
+     VIEWER_STDOUT_SELF - standard output (default)
-     VIEWER_STDOUT_WORLD - synchronized standard
         output where only the first processor opens
         the file.  All other processors send their 
         data to the first processor to print. 

   The user can open alternative vistualization contexts with
+    ViewerFileOpenASCII() - Outputs vector to a specified file
.    ViewerFileOpenBinary() - Outputs vector in binary to a
         specified file; corresponding input uses VecLoad()
.    ViewerDrawOpenX() - Outputs vector to an X window display
-    ViewerMatlabOpen() - Outputs vector to Matlab viewer

.keywords: view, visualize, output, print, write, draw

.seealso: DFShellGetInfo(), VecView()
@*/
int DFVecView(DFVec v,Viewer viewer)
{  
  ViewerType vtype;
  int        ierr, rank, size, format;
  FILE       *fd;
  DF         df;

  PetscFunctionBegin;
  if (!viewer) {
    viewer = VIEWER_STDOUT_SELF;
  }
  ierr = DFVecGetDFShell(v,&df); CHKERRQ(ierr);
  ierr = ViewerGetType(viewer,&vtype); CHKERRQ(ierr);
  if (vtype == ASCII_FILE_VIEWER || vtype == ASCII_FILES_VIEWER) {
    ierr = ViewerGetFormat(viewer,&format); CHKERRQ(ierr);
    MPI_Comm_rank(df->comm,&rank);
    MPI_Comm_size(df->comm,&size);
    ierr = ViewerASCIIGetPointer(viewer,&fd); CHKERRQ(ierr);
    if (format == VIEWER_FORMAT_ASCII_INFO || format == VIEWER_FORMAT_ASCII_INFO_LONG) {
      PetscFPrintf(df->comm,fd,"DF Context:\n");
      PetscFPrintf(df->comm,fd,"  type=%d, dim=%d, nc=%d, ord=%d, global: n0=%d, n1=%d, n2=%d\n",
       df->type,df->dim,df->nc,df->order,df->gnp[0],df->gnp[1],df->gnp[2]);
      if (size > 1) {
        PetscSequentialPhaseBegin(df->comm,1);
        fprintf(fd,"  [%d] local: n0=%d, n1=%d, n2=%d\n",rank,df->lnp[0],df->lnp[1],df->lnp[2]);
        PetscSequentialPhaseEnd(df->comm,1);
      }
    } else { /* View discrete function */
      if (df->ops->view) {ierr = (*df->ops->view)(v,viewer); CHKERRQ(ierr);}
    }
  }
  PetscFunctionReturn(0);
}

