#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: regdf.c,v 1.38 1998/03/16 18:15:20 balay Exp $";
#endif

/*
   Implements sequential discrete function utilities for regular grids.
*/
#include "src/dfvec/dfvimpl.h" 
#include "src/vec/vecimpl.h"

#undef __FUNC__  
#define __FUNC__ "DFVecRefineBase_SeqRegular_Private"
static int DFVecRefineBase_SeqRegular_Private(Vec v,DF df,Vec v2,DF df2)
{
  int    i, j, k, ierr, mx, my, mz, mx2, my2;
  int    zk, zk1, zk2, yj, yj1, yj2, zkyj, zkyj1, zkyj2;
  Scalar *va, *va2;

  mx  = df->gnp[0];
  my  = df->gnp[1];
  mz  = df->gnp[2];
  mx2 = df2->gnp[0];
  my2 = df2->gnp[1];

  ierr = VecGetArray(v,&va);   CHKERRQ(ierr);
  ierr = VecGetArray(v2,&va2); CHKERRQ(ierr);

  /* Compute new vector values. For now, assume ordering by x most rapidly,
     then y, z */
  if (df->dim == 1) {
    va2[0] = va[0];
    for (i=1; i<mx; i++) {
      va2[2*i]   = va[i];
      va2[2*i-1] = 0.5*(va[i] + va[i-1]);
    }
  }
  else if (df->dim == 2) {
    for (j=0; j<my; j++) {
      yj = j*mx; yj2 = j*mx2*2;
      for (i=0; i<mx; i++)
        va2[yj2 + i*2] = va[yj + i]; 
    }
    for (j=0; j<my; j++) {
      yj = j*mx; yj2 = j*mx2*2;
      for (i=0; i<mx-1; i++)
        va2[yj2 + i*2+1] = (va[yj + i] + va[yj + i+1]) * 0.5;
    }
    for (j=0; j<my-1; j++) {
      yj = j*mx; yj1 = (j+1)*mx; yj2 = (j*2+1)*mx2;
      for (i=0; i<mx; i++)
        va2[yj2 + i*2] = (va[yj + i] + va[yj1 + i]) * 0.5; 
    }
    for (j=0; j<my-1; j++) {
      yj = j*mx; yj1 = (j+1)*mx; yj2 = (j*2+1)*mx2;
      for (i=0; i<mx-1; i++)
        va2[yj2 + i*2+1] = 0.25 * (va[yj + i] + va[yj + i+1] 
                             + va[yj1 + i] + va[yj1 + i+1]);
    }
  }
  else if (df->dim == 3) {
    /* original z points */
    for (k=0; k<mz; k++) {
      zk = k*mx*my; zk2 = k*mx2*my2*2;
      for (j=0; j<my; j++) {
        zkyj = zk + j*mx; zkyj2 = zk2 + j*mx2*2;
        for (i=0; i<mx; i++)
          va2[zkyj2 + i*2] = va[zkyj + i]; 
      }
      for (j=0; j<my; j++) {
        zkyj = zk + j*mx; zkyj2 = zk2 + j*mx2*2;
        for (i=0; i<mx-1; i++)  
          va2[zkyj2 + i*2+1] = (va[zkyj + i] + va[zkyj + i+1]) * 0.5;
      }
      for (j=0; j<my-1; j++) {
        zkyj = zk + j*mx; zkyj1 = zk + (j+1)*mx; zkyj2 = zk2 + (j*2+1)*mx2;
        for (i=0; i<mx; i++) 
          va2[zkyj2 + i*2] = (va[zkyj + i] + va[zkyj1 + i]) * 0.5; 
      }
      for (j=0; j<my-1; j++) {
        zkyj = zk + j*mx; zkyj1 = zk + (j+1)*mx; zkyj2 = zk2 + (j*2+1)*mx2;
        for (i=0; i<mx-1; i++)
          va2[zkyj2 + i*2+1] = 
             0.25 * (va[zkyj + i] + va[zkyj + i+1] + va[zkyj1 + i] + va[zkyj1 + i+1]);
      }
    }
    /* intermediate z points, assuming above have been computed */
    for (k=0; k<mz-1; k++) {
      zk = k*2*mx2*my2; zk1 = (k*2+1)*mx2*my2; zk2 = (k*2+2)*mx2*my2;
      for (j=0; j<my2; j++) {
        zkyj = zk + j*mx2; zkyj1 = zk1 + j*mx2; zkyj2 = zk2 + j*mx2;
        for (i=0; i<mx2; i++)
          va2[zkyj1 + i] = (va2[zkyj + i] + va2[zkyj2 + i]) * 0.5; 
      }
    }
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecRefineVector_SeqRegular"
static int DFVecRefineVector_SeqRegular(DFVec v,DFVec *vref)
{
  DF  df, dfr;
  int ierr, gsize, i, nsub, nrsub;
  Vec vr, *vsub, *vrsub;

  ierr = DFVecGetDFShell(v,&df); CHKERRQ(ierr);
  df->rset = 1;
  ierr = DFShellDuplicate(df,&dfr); CHKERRQ(ierr);
  dfr->gnp[0] = dfr->lnp[0] = 2*df->gnp[0] - 1;
  dfr->gnp[1] = dfr->lnp[1] = 2*df->gnp[1] - 1;
  dfr->gnp[2] = dfr->lnp[2] = 2*df->gnp[2] - 1;
  gsize = dfr->gnp[0] * dfr->gnp[1] * dfr->gnp[2] * df->nc;
  ierr = VecCreateSeq(PETSC_COMM_SELF,gsize,&vr); CHKERRQ(ierr); 
  ierr = DFVecShellAssociate(dfr,vr); CHKERRQ(ierr);
  *vref = vr;
  if (df->nc == 1) {
    ierr = DFVecRefineBase_SeqRegular_Private(v,df,vr,dfr); CHKERRQ(ierr);
  }
  else { /* Refine component vectors */
    /* Could make more efficient by doing all at once instead of 1 at a time */
    ierr = DFVecGetComponentVectors(v,&nsub,&vsub); CHKERRQ(ierr);
    ierr = DFVecGetComponentVectors(vr,&nrsub,&vrsub); CHKERRQ(ierr);
    for (i=0; i<df->nc; i++) {
      ierr = DFVecRefineBase_SeqRegular_Private(vsub[i],df,vrsub[i],dfr); CHKERRQ(ierr);
    }
    ierr = DFVecAssembleFullVector(vrsub,vr); CHKERRQ(ierr);
    ierr = VecDestroyVecs(vrsub,dfr->nc); CHKERRQ(ierr);
    ierr = VecDestroyVecs(vsub,df->nc); CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecDrawTensorContours_SeqRegular"
/*
   DFVecDrawTensorContours_SeqRegular - Draws contour plot of each component within a
   multi-component vector.
  
   Notes:
   Currently this routine is supports 2D structured problems only, where
   DrawTensorContour() is called for each component of the vector.

   If DFShellSetCoordinates() has not been called to set the grid 
   coordinates, a uniform grid over [0,1] X [0,1] is used for the
   contour plot.
*/
static int DFVecDrawTensorContours_SeqRegular(DFVec v,Viewer *viewer)
{
  int  ierr, i,  nc;
  Vec  *vsub;
  Draw draw;
  DF   df;

  ierr = DFVecGetComponentVectors(v,&nc,&vsub); CHKERRQ(ierr);
  PLogObjectParents(v,nc,vsub);
  ierr = DFVecGetDFShell(v,&df); CHKERRQ(ierr);
  for (i=0; i<nc; i++) {
    ierr = ViewerDrawGetDraw(viewer[i],&draw);CHKERRQ(ierr);
    ierr = DrawTensorContour(draw,df->gnp[0],df->gnp[1],df->coord[0],
                             df->coord[1],vsub[i]); CHKERRQ(ierr);
    ierr = DrawSynchronizedFlush(draw); CHKERRQ(ierr);
  }
  ierr = VecDestroyVecs(vsub,nc); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecDrawTensorContoursX_SeqRegular"
/*
   DFVecDrawTensorContoursX_SeqRegular - Draws contour plot of each component within a
   multi-component vector.
  
   Notes:
   Currently this routine is supports 2D structured problems only, where
   DrawTensorContour() is called for each component of the vector.

   If DFShellSetCoordinates() has not been called to set the grid 
   coordinates, a uniform grid over [0,1] X [0,1] is used for the
   contour plot.
*/
static int DFVecDrawTensorContoursX_SeqRegular(DFVec v,int width,int height)
{
  int  ierr, i, nc;
  Vec  *vsub;
  Draw *draw;
  DF   df;

  ierr = DFVecGetComponentVectors(v,&nc,&vsub); CHKERRQ(ierr);
  PLogObjectParents(v,nc,vsub);
  draw = (Draw *) PetscMalloc(nc*sizeof(Draw *));
  ierr = DFVecGetDFShell(v,&df); CHKERRQ(ierr);
  for (i=0; i<nc; i++) {
    ierr = DrawOpenX(v->comm,0,df->label[i],PETSC_DECIDE,PETSC_DECIDE,width,height,&draw[i]); CHKERRQ(ierr);
    PLogObjectParent(df,draw[i]);
    ierr = DrawTensorContour(draw[i],df->gnp[0],df->gnp[1],df->coord[0],
                             df->coord[1],vsub[i]); CHKERRQ(ierr);
    ierr = DrawSynchronizedFlush(draw[i]); CHKERRQ(ierr);
    ierr = DrawPause(draw[i]); CHKERRQ(ierr);
  }
  for (i=0; i<nc; i++) {ierr = DrawDestroy(draw[i]); CHKERRQ(ierr);}
  ierr = VecDestroyVecs(vsub,nc); CHKERRQ(ierr);
  PetscFree(draw);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecDrawTensorSurfaceContoursVRML_SeqRegular"
/*
   DFVecDrawTensorContoursX_SeqRegular - Draws contour plot of each component within a
   multi-component vector.
  
   Notes:
   Currently this routine supports 2D and 3D structured problems only, where
   DrawTensorSurfaceContour_VRML() is called for each component of the vector.
   For 3D problems, the data is viewed in layers in the z-direction. We
   eventually will generalize this.

   If DFShellSetCoordinates() has not been called to set the grid 
   coordinates, a uniform grid over [0,1] X [0,1] is used for the
   contour plot.
*/

static int DFVecDrawTensorSurfaceContoursVRML_SeqRegular(DFVec v)
{
  /* #if !defined(USE_PETSC_COMPLEX) */
#if defined(foo)
  int    ierr, flg, i, k, nc, m, n, nocontour;
  Vec    *vsub;
  Draw   *draw;
  DF     df;
  Scalar *va, *vak;
  double *c0, *c1, h;
  char   filebase[1024], filename[1024], strnum[64];

  ierr = DFVecGetComponentVectors(v,&nc,&vsub); CHKERRQ(ierr);
  PLogObjectParents(v,nc,vsub);
  draw = (Draw *) PetscMalloc(nc*sizeof(Draw *));
  ierr = DFVecGetDFShell(v,&df); CHKERRQ(ierr);

  /* Fill up x and y coordinates */
  if (df->cset) {
    c0 = df->coord[0];
    c1 = df->coord[1];   
  } else {
    m = df->gnp[0]; n = df->gnp[1];
    c0 = (double *) PetscMalloc( (m+n)*sizeof(double) ); CHKPTRQ(c0);
    c1 = c0 + m;
    h = 1.0/(m-1);
    c0[0] = 0.0;
    for ( i=1; i<m; i++ ) c0[i] = c0[i-1] + h;
    h = 1.0/(n-1);
    c1[0] = 0.0;
    for ( i=1; i<n; i++ ) c1[i] = c1[i-1] + h;
  }

  /* Output file */
  ierr = OptionsGetString(PETSC_NULL,"-dfvec_base_vrmlfile",filebase,1000,&flg); CHKERRQ(ierr);
  ierr = OptionsHasName(PETSC_NULL,"-dfvec_vrml_nocontour",&nocontour);
  if (!flg) PetscStrcpy(filebase,"dfvec");
  for (i=0; i<nc; i++) {
    ierr = VecGetArray(vsub[i],&va); CHKERRQ(ierr);
    vak  = va;
    sprintf(strnum,".%d.vrml",i);
    PetscStrcpy(filename,filebase);
    PetscStrcat(filename,strnum);
    ierr = DrawOpenVRML(v->comm,filename,df->label[i],&draw[i]); CHKERRQ(ierr);
    PLogObjectParent(df,draw[i]);
    for (k=0; k<df->gnp[2]; k++) {
      ierr = DrawBOP(draw[i]); CHKERRQ(ierr);
      if (nocontour) {
        ierr = DrawTensorSurface_VRML(draw[i],c0,df->gnp[0],
                 c1,df->gnp[1],vak,64); CHKERRQ(ierr);
      } else {
        ierr = DrawTensorSurfaceContour_VRML(draw[i],c0,df->gnp[0],
                 c1,df->gnp[1],vak,64); CHKERRQ(ierr);
      }
      ierr = DrawEOP(draw[i]); CHKERRQ(ierr);
      vak += df->gnp[0]*df->gnp[1];
    }
    ierr = VecRestoreArray(vsub[i],&va); CHKERRQ(ierr);
  }
  for (i=0; i<nc; i++) {ierr = DrawDestroy(draw[i]); CHKERRQ(ierr);}
  if (!df->cset) PetscFree(c0);
  ierr = VecDestroyVecs(vsub,nc); CHKERRQ(ierr);
  PetscFree(draw);
#endif
  PetscFunctionReturn(0);
}

extern int DFVecAssembleFullVector_SeqGeneral(Vec*,DFVec);

static struct _DFOps DFSeqRegularOps = 
   {DFVecRefineVector_SeqRegular,
    DFRefineCoordinates_SeqRegular,
    DFVecAssembleFullVector_SeqGeneral,
    DFVecDrawTensorContoursX_SeqRegular,
    DFVecDrawTensorContours_SeqRegular,
    DFVecDrawTensorSurfaceContoursVRML_SeqRegular,
    DFCopy_SeqRegular};

#undef __FUNC__  
#define __FUNC__ "DFSetSeqRegularOps_Private"
int DFSetSeqRegularOps_Private(DF v)
{
  PetscMemcpy(v->ops,&DFSeqRegularOps,sizeof(DFSeqRegularOps));
  PetscFunctionReturn(0);
}
