#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: regmpidf.c,v 1.87 1998/04/09 04:17:38 bsmith Exp $";
#endif

/*
   Implements parallel discrete function utilities for regular grids.
*/
#include "src/dfvec/dfvimpl.h" 
#include "src/vec/vecimpl.h"
#include "pinclude/pviewer.h"
#include "src/da/daimpl.h"

#undef __FUNC__  
#define __FUNC__ "DFVecRefineBase_MPIRegular_Private"
/*
   DFVecRefineBase_MPIRegular_Private - Refines single component vector.

   Input Parameters:
   lv - local ghosted vector

   Output Parameters:
   lv2 - refined local (unghosted) vector
 */
static int DFVecRefineBase_MPIRegular_Private(Vec lv,DA da,Vec lv2,DA da2)
{
  int    i, j, k, ierr, M, N, P, xe1, ye1, xs, xm, xe, ys, ym, ye;
  int    Xs, Xm, Ys, Ym, Zs, Zm, xs1, dim;
  int    zk2, yj2, zkyj2, zs, zm, ze, ze1, ze2;
  int    xs2, xm2, xe2, ys2, ym2, ye2, zs2, zm2, ys1, zs1;
  Scalar *va, *va2;

  /* Refine base vector. For now, assume ordering by x most rapidly, then y, z */
  ierr = DAGetCorners(da,&xs,&ys,&zs,&xm,&ym,&zm); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(da,&Xs,&Ys,&Zs,&Xm,&Ym,&Zm); CHKERRQ(ierr);
  xe = xs+xm; ye = ys+ym; ze = zs+zm;
  ierr = DAGetInfo(da,&dim,&M,&N,&P,PETSC_NULL,PETSC_NULL,PETSC_NULL,
         PETSC_NULL,PETSC_NULL,PETSC_NULL); CHKERRQ(ierr);
  ierr = DAGetCorners(da2,&xs2,&ys2,&zs2,&xm2,&ym2,&zm2); CHKERRQ(ierr);
  xe2 = xs2+xm2; ye2 = ys2+ym2; ze2 = zs2+zm2;
  ierr = VecGetArray(lv,&va);   CHKERRQ(ierr);
  ierr = VecGetArray(lv2,&va2); CHKERRQ(ierr);

#define x_va(i,j) va[(i)-Xs]
#define xy_va(i,j) va[(i)-Xs + ((j)-Ys)*Xm]
#define xyz_va(i,j,k) va[(i)-Xs + ((j)-Ys)*Xm + ((k)-Zs)*Xm*Ym]

  /* These shifts determine whether extra starting/ending ghost points
     on processor boundaries need to be processed for refinement */
  if (xe2 == 2*xe) xe1 = xe;
  else             xe1 = xe-1;
  if (ye2 == 2*ye) ye1 = ye;
  else             ye1 = ye-1;
  if (ze2 == 2*ze) ze1 = ze;
  else             ze1 = ze-1;
  if (xs2 == 2*xs) xs1 = xs;
  else             xs1 = xs-1;
  if (ys2 == 2*ys) ys1 = ys;
  else             ys1 = ys-1;
  if (zs2 == 2*zs) zs1 = zs;
  else             zs1 = zs-1;
  if (dim == 1) {
    for (i=xs; i<xe; i++) {
      va2[2*i-xs2]   = x_va(i,j);
    }
    for (i=xs1; i<xe1; i++) {
      va2[i*2+1-xs2] = 0.5 * (x_va(i,j) + x_va(i+1,j));
    }
  }
  else if (dim == 2) {
    for (j=ys; j<ye; j++) {
      yj2 = (2*j-ys2)*xm2;
      for (i=xs; i<xe; i++) {
        va2[yj2 + i*2-xs2] = xy_va(i,j);
      } 
    }
    for (j=ys; j<ye; j++) {
      yj2 = (2*j-ys2)*xm2;
      for (i=xs1; i<xe1; i++) {
        va2[yj2 + i*2+1-xs2] = 0.5 * (xy_va(i,j) + xy_va(i+1,j));
      } 
    }
    for (j=ys1; j<ye1; j++) {
      yj2 = (2*j-ys2+1)*xm2;
      for (i=xs; i<xe; i++) {
        va2[yj2 + i*2-xs2] = 0.5 * (xy_va(i,j) + xy_va(i,j+1));
      } 
    }
    for (j=ys1; j<ye1; j++) {
      yj2 = (2*j-ys2+1)*xm2;
      for (i=xs1; i<xe1; i++) {
        va2[yj2 + i*2+1-xs2] = 0.25 * (xy_va(i,j) + xy_va(i+1,j) + xy_va(i,j+1) + xy_va(i+1,j+1));
      } 
    } 
  }

  else if (dim == 3) {
    for (k=zs; k<ze; k++) {
      /* original z points */
      zk2 = (2*k-zs2)*xm2*ym2;
      for (j=ys; j<ye; j++) {
        zkyj2 = zk2 + (2*j-ys2)*xm2;
        for (i=xs; i<xe; i++) 
          va2[zkyj2 + i*2-xs2] = xyz_va(i,j,k);
      }
      for (j=ys; j<ye; j++) {
        zkyj2 = zk2 + (2*j-ys2)*xm2;
        for (i=xs1; i<xe1; i++)
          va2[zkyj2 + i*2+1-xs2] = 0.5 * (xyz_va(i,j,k) + xyz_va(i+1,j,k));
      }
      for (j=ys1; j<ye1; j++) {
        zkyj2 = zk2 + (2*j-ys2+1)*xm2;
        for (i=xs; i<xe; i++)
          va2[zkyj2 + i*2-xs2] = 0.5 * (xyz_va(i,j,k) + xyz_va(i,j+1,k));
      }
      for (j=ys1; j<ye1; j++) {
        zkyj2 = zk2 + (2*j-ys2+1)*xm2;
        for (i=xs1; i<xe1; i++) 
          va2[zkyj2 + i*2+1-xs2] = 0.25 * (xyz_va(i,j,k) + xyz_va(i+1,j,k)
                                   + xyz_va(i,j+1,k) + xyz_va(i+1,j+1,k));
      }
    }

    /* intermediate z points */
    for (k=zs1; k<ze1; k++) {
      zk2 = (2*k+1-zs2)*xm2*ym2;
      for (j=ys; j<ye; j++) {
        zkyj2 = zk2 + (2*j-ys2)*xm2;
        for (i=xs; i<xe; i++) 
          va2[zkyj2 + i*2-xs2] = 0.5 * (xyz_va(i,j,k) + xyz_va(i,j,k+1));
      }
      for (j=ys; j<ye; j++) {
        zkyj2 = zk2 + (2*j-ys2)*xm2;
        for (i=xs1; i<xe1; i++)
          va2[zkyj2 + i*2+1-xs2] = 0.25 * (xyz_va(i,j,k) + xyz_va(i+1,j,k)
                                         + xyz_va(i,j,k+1) + xyz_va(i+1,j,k+1));
      }
      for (j=ys1; j<ye1; j++) {
        zkyj2 = zk2 + (2*j-ys2+1)*xm2;
        for (i=xs; i<xe; i++)
          va2[zkyj2 + i*2-xs2] = 0.25 * (xyz_va(i,j,k) + xyz_va(i,j+1,k)
                                         + xyz_va(i,j,k+1) + xyz_va(i,j+1,k+1));
      }
      for (j=ys1; j<ye1; j++) {
        zkyj2 = zk2 + (2*j-ys2+1)*xm2;
        for (i=xs1; i<xe1; i++) 
          va2[zkyj2 + i*2+1-xs2] = 0.125 * (xyz_va(i,j,k) + xyz_va(i,j+1,k)
                                          + xyz_va(i,j,k+1) + xyz_va(i,j+1,k+1)
                                          + xyz_va(i+1,j,k) + xyz_va(i+1,j+1,k)
                                          + xyz_va(i+1,j,k+1) + xyz_va(i+1,j+1,k+1));
      }
    }
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecRefineVectorSetUp_MPIRegular_Private"
/*
   DFVecRefineVectorSetUp_MPIRegular_Private - Sets up refinement.

   Input Parameters:
.  df - grid info context, formed by DFVecCreate()
.  da - distributed array (parallel, structured case only)
*/
static int DFVecRefineVectorSetUp_MPIRegular_Private(DF df)
{
  int      ierr, M, N, P, m, n, p, w, dim;
  MPI_Comm comm;

  if (df->rset) PetscFunctionReturn(0); /* Do nothing if refinement info has been set */
  ierr = DAGetInfo(df->da_user,&dim,&M,&N,&P,&m,&n,&p,&w,PETSC_NULL,PETSC_NULL); CHKERRQ(ierr);
  if (dim != df->dim || w != df->nc) {
    SETERRQ(PETSC_ERR_ARG_INCOMP,0,"Incompatible DA and DF contexts");
  }
  comm = df->comm;

  /* Create new internal DA, width = 1, to support refinement. We assume that
     DACreate() forms a new DA with the same distribution as the input DA.
   */
  if (dim == 1) {
    ierr = DACreate1d(comm,DA_NONPERIODIC,M,w,1,PETSC_NULL,&df->da); CHKERRQ(ierr);
  } else if (dim == 2) {
    ierr = DACreate2d(comm,DA_NONPERIODIC,DA_STENCIL_BOX,M,N,m,n,w,1,0,0,&df->da);CHKERRQ(ierr);
  } else if (dim == 3) {
    ierr = DACreate3d(comm,DA_NONPERIODIC,DA_STENCIL_BOX,M,N,P,m,n,p,w,1,0,0,0,&df->da);CHKERRQ(ierr);
  } else SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"only dimensions 1,2,3 supported");
  PLogObjectParent(df,df->da);
  df->rset = 1;
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecRefineVector_MPIRegular"
int DFVecRefineVector_MPIRegular(Vec vin,Vec *vref)
{
  DF    df, dfr, dfr2;
  Vec   v, vr, vrlocal, *vsub, *vrsub;
  int   ierr, i, nsub, nrsub;
  DFVec vloc, vr2;
  DA    daref;

  ierr = DFVecGetDFShell(vin,&df); CHKERRQ(ierr);
  if (!df->rset) {ierr = DFVecRefineVectorSetUp_MPIRegular_Private(df); CHKERRQ(ierr);}

  /* Get ghost points */
  ierr = DACreateLocalVector(df->da,&vloc); CHKERRQ(ierr);
  ierr = DACreateGlobalVector(df->da,&v); CHKERRQ(ierr);
  ierr = VecCopy(vin,v); CHKERRQ(ierr);
  ierr = DAGlobalToLocalBegin(df->da,v,INSERT_VALUES,vloc); CHKERRQ(ierr);
  ierr = DAGlobalToLocalEnd(df->da,v,INSERT_VALUES,vloc); CHKERRQ(ierr);

  /* Refine internal DA */
  ierr = DARefine(df->da,&daref); CHKERRQ(ierr);
  ierr = DACreateGlobalVector(daref,&vr); CHKERRQ(ierr);
  ierr = DFVecGetDFShell(vr,&dfr); CHKERRQ(ierr);

  if (df->nc == 1) {
    ierr = DFVecRefineBase_MPIRegular_Private(vloc,df->da,vr,daref); CHKERRQ(ierr);
  }
  else { /* Refine component vectors */
    /* Could make more efficient by doing all at once instead of 1 at a time */
    ierr = DFVecGetComponentVectors(vloc,&nsub,&vsub); CHKERRQ(ierr);
    ierr = DFVecGetComponentVectors(vr,&nrsub,&vrsub); CHKERRQ(ierr);
    PLogObjectParents(vloc,nsub,vsub); PLogObjectParents(vr,nrsub,vrsub);
    for (i=0; i<nsub; i++) {
      ierr = DFVecRefineBase_MPIRegular_Private(vsub[i],df->da,vrsub[i],daref); CHKERRQ(ierr);
    }
    ierr = DFVecAssembleFullVector(vrsub,vr); CHKERRQ(ierr);
    ierr = VecDestroyVecs(vrsub,nrsub); CHKERRQ(ierr);
    ierr = VecDestroyVecs(vsub,nsub); CHKERRQ(ierr);
  }

  /* Note that we copy and destroy the new vector, so that it can be destroyed
     before its parent DA; otherwise, this creates problems with logging. */
  ierr = VecDuplicate(vr,&vr2); CHKERRQ(ierr);
  ierr = DFShellDuplicate(dfr,&dfr2); CHKERRQ(ierr);

  /* Really should clear the first DF shell from the vector child so we can
     associate the real DF */
  ierr = VecCopy(vr,vr2); CHKERRQ(ierr);
  ierr = DFVecShellAssociate(dfr2,vr2); CHKERRQ(ierr);
  ierr = DFShellDestroy(dfr2);  CHKERRQ(ierr); 
  *vref = vr2;

  /* Free work space */
  /* We must explicitly extract the local vector so it can be destroyed; 
     this will change in the near future */
  ierr = DACreateLocalVector(daref,&vrlocal); CHKERRQ(ierr);
  ierr = VecDestroy(vrlocal); CHKERRQ(ierr);

  ierr = VecDestroy(vloc); CHKERRQ(ierr);
  ierr = VecDestroy(v); CHKERRQ(ierr);     ierr = VecDestroy(vr); CHKERRQ(ierr);
  ierr = DADestroy(df->da); CHKERRQ(ierr); ierr = DADestroy(daref); CHKERRQ(ierr);

  PetscFunctionReturn(0);
}

/* 
   DFVecFormUniVec_MPIRegular_Private - Forms a local vector on processor 0 with
   the mapped ordering; no other processors form a local vector, but instead just
   get vout = PETSC_NULL.
 */
#undef __FUNC__  
#define __FUNC__ "DFVecFormUniVec_MPIRegular_Private"
int DFVecFormUniVec_MPIRegular_Private(DFVec dfv,Vec *vout)
{
  Vec        v, vin = (Vec) dfv;
  DF         df;
  int        i, rank, len, work, nloc, n, j, size, ierr, *gtog1;
  int        *starts, start, end, N;
  MPI_Status status;
  Scalar     *values, *v_array;

  ierr = DFVecGetDFShell(dfv,&df); CHKERRQ(ierr);
  ierr = DAGetGlobalToGlobal1_Private(df->da_user,&gtog1); CHKERRQ(ierr);
  ierr = VecGetOwnershipRange(vin,&start,&end); CHKERRQ(ierr);
  ierr = VecGetArray(vin,&v_array); CHKERRQ(ierr);
  ierr = VecGetLocalSize(vin,&nloc); CHKERRQ(ierr);
  work = nloc;
  ierr = MPI_Reduce(&work,&len,1,MPI_INT,MPI_MAX,0,df->comm);CHKERRQ(ierr);
  MPI_Comm_size(df->comm,&size);
  starts = (int *) PetscMalloc( size*sizeof(int) ); CHKPTRQ(starts);
  ierr = MPI_Allgather(&start,1,MPI_INT,starts,1,MPI_INT,df->comm);CHKERRQ(ierr);
  MPI_Comm_rank(df->comm,&rank);
  if (!rank) {
    ierr = VecGetSize(vin,&N); CHKERRQ(ierr);
    ierr = VecCreateSeq(PETSC_COMM_SELF,N,&v); CHKERRQ(ierr);
    values = (Scalar *) PetscMalloc( len*sizeof(Scalar) ); CHKPTRQ(values);
    for ( i=0; i<nloc; i++ ) {
      ierr = VecSetValues(v,1,&gtog1[i],&v_array[i],INSERT_VALUES); CHKERRQ(ierr);
    }
    /* receive values and insert into local vector */
    for ( j=1; j<size; j++ ) {
      ierr = MPI_Recv(values,len,MPIU_SCALAR,j,47,df->comm,&status);CHKERRQ(ierr);
      ierr = MPI_Get_count(&status,MPIU_SCALAR,&n);CHKERRQ(ierr);          
      for ( i=0; i<n; i++ ) {
        ierr = VecSetValues(v,1,&gtog1[i+starts[j]],&values[i],INSERT_VALUES); CHKERRQ(ierr);
      }          
    }
    PetscFree(values);
    *vout = v;
  } else {
    /* send values */
    ierr = MPI_Send(v_array,nloc,MPIU_SCALAR,0,47,df->comm);CHKERRQ(ierr);
    ierr = VecCreateSeq(PETSC_COMM_SELF,0,vout); CHKERRQ(ierr);
  }
  PetscFree(starts);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecView_MPIRegular"
static int DFVecView_MPIRegular(DFVec dfv,Viewer viewer)
{
  Vec        v, vin = (Vec) dfv;
  DF         df;
  int        rank, size, ierr;

  if (!viewer) { /* so that viewers may be used from debuggers */
    viewer = VIEWER_STDOUT_SELF;
  }

  ierr = DFVecGetDFShell(dfv,&df); CHKERRQ(ierr);
  MPI_Comm_size(df->comm,&size);
  if (size == 1) {  /* Call regular vector viewer if using just 1 processor */
    ierr = VecView(vin,viewer); CHKERRQ(ierr);
  } else {
    /* Form a local vector on processor 0 with the mapped ordering 
       Eventually we should work with sections of the vector, 1 at a time,
       instead of assembling the full parallel vector on 1 processor. */
    ierr = DFVecFormUniVec_MPIRegular_Private(dfv,&v); CHKERRQ(ierr);
    MPI_Comm_rank(df->comm,&rank);
    if (rank == 0) {
      ierr = VecView(v,viewer); CHKERRQ(ierr);
    }
    ierr = VecDestroy(v); CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecDrawTensorContoursX_MPIRegular"
/*
   DFVecDrawTensorContoursX_MPIRegular - Draws contour plot of each component within a
   multi-component vector.
  
   Notes:
   Currently this routine is supports 2D structured problems only, where
   DrawTensorContour() is called for each component of the vector.

   If DFShellSetCoordinates() has not been called to set the grid 
   coordinates, a uniform grid over [0,1] X [0,1] is used for the
   contour plot.
*/
static int DFVecDrawTensorContoursX_MPIRegular(DFVec dfv,int width,int height)
{
  int  ierr, i, nc, rank;
  Vec  *vsub, v;
  Draw *draw;
  DF   df, df_local;
  DFVec dfv1;

  /* Form a local vector on processor 0 with the mapped ordering 
     Eventually we should work with sections of the vector, 1 at a time,
     instead of assembling the full parallel vector on 1 processor. */
  ierr = DFVecFormUniVec_MPIRegular_Private(dfv,&v); CHKERRQ(ierr);
  ierr = DFVecGetDFShell(dfv,&df); CHKERRQ(ierr);

  MPI_Comm_rank(df->comm,&rank);
  if (rank == 0) {
    ierr = DFShellGetLocalDFShell(df,&df_local);
    ierr = DFVecShellAssociate(df_local,v); CHKERRQ(ierr);
    dfv1 = (DFVec) v;
    ierr = DFVecGetComponentVectors(dfv1,&nc,&vsub); CHKERRQ(ierr);
    PLogObjectParents(dfv,nc,vsub);
    draw = (Draw *) PetscMalloc(nc*sizeof(Draw *));
    for (i=0; i<nc; i++) {
      ierr = DrawOpenX(v->comm,0,df->label[i],PETSC_DECIDE,PETSC_DECIDE,width,height,&draw[i]); CHKERRQ(ierr);
      PLogObjectParent(df,draw[i]);
      ierr = DrawTensorContour(draw[i],df->gnp[0],df->gnp[1],PETSC_NULL,
                               PETSC_NULL,vsub[i]); CHKERRQ(ierr);
       /*      ierr = DrawTensorContour(draw[i],df->gnp[0],df->gnp[1],df->coord[0],
                               df->coord[1],vsub[i]); CHKERRQ(ierr); */
      ierr = DrawSynchronizedFlush(draw[i]); CHKERRQ(ierr);
      ierr = DrawPause(draw[i]); CHKERRQ(ierr);
    }
    for (i=0; i<nc; i++) {ierr = DrawDestroy(draw[i]); CHKERRQ(ierr);}
    ierr = VecDestroyVecs(vsub,nc); CHKERRQ(ierr);
    ierr = VecDestroy(v); CHKERRQ(ierr);
    PetscFree(draw);
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecDrawTensorContours_MPIRegular"
/*
   DFVecDrawTensorContours_MPIRegular - Draws contour plot of each component within a
   multi-component vector.
  
   Notes:
   Currently this routine is supports 2D structured problems only, where
   DrawTensorContour() is called for each component of the vector.

   If DFShellSetCoordinates() has not been called to set the grid 
   coordinates, a uniform grid over [0,1] X [0,1] is used for the
   contour plot.
*/
static int DFVecDrawTensorContours_MPIRegular(DFVec dfv,Viewer *viewer)
{
  int  ierr, i, nc, rank;
  Vec  *vsub, v;
  Draw draw;
  DF   df, df_local;
  DFVec dfv1;

  /* Form a local vector on processor 0 with the mapped ordering 
     Eventually we should work with sections of the vector, 1 at a time,
     instead of assembling the full parallel vector on 1 processor. */
  ierr = DFVecFormUniVec_MPIRegular_Private(dfv,&v); CHKERRQ(ierr);
  ierr = DFVecGetDFShell(dfv,&df); CHKERRQ(ierr);

  MPI_Comm_rank(df->comm,&rank);
  /* if (rank == 0) { */
    ierr = DFShellGetLocalDFShell(df,&df_local);
    ierr = DFVecShellAssociate(df_local,v); CHKERRQ(ierr);
    dfv1 = (DFVec) v;
    ierr = DFVecGetComponentVectors(dfv1,&nc,&vsub); CHKERRQ(ierr);
    PLogObjectParents(dfv,nc,vsub);
    for (i=0; i<nc; i++) {
      ierr = ViewerDrawGetDraw(viewer[i],&draw); CHKERRQ(ierr);
      ierr = DrawTensorContour(draw,df->gnp[0],df->gnp[1],PETSC_NULL,
                               PETSC_NULL,vsub[i]); CHKERRQ(ierr);
       /*      ierr = DrawTensorContour(draw[i],df->gnp[0],df->gnp[1],df->coord[0],
                               df->coord[1],vsub[i]); CHKERRQ(ierr); */
      ierr = DrawSynchronizedFlush(draw); CHKERRQ(ierr);
    }
    ierr = VecDestroyVecs(vsub,nc); CHKERRQ(ierr);
    ierr = VecDestroy(v); CHKERRQ(ierr);
    /* } */
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFVecDrawTensorSurfaceContoursVRML_MPIRegular"
/*
   DFVecDrawTensorSurfaceContoursVRML_MPIRegular - Draws contour plot of each
   component within a multi-component vector.
  
   Notes:
   Currently this routine supports 2D and 3D structured problems only, where
   DrawTensorContour() is called for each component of the vector.  For
   3D problems, the domain is sliced by z-components.  We should generalize
   this eventually.

   If DFShellSetCoordinates() has not been called to set the grid 
   coordinates, a uniform grid over [0,1] X [0,1] is used for the
   contour plot.
*/
static int DFVecDrawTensorSurfaceContoursVRML_MPIRegular(DFVec dfv)
{
#if defined(foo)
  /* #if !defined(USE_PETSC_COMPLEX) */
  int    ierr, i, k, m, nc, n, rank, flg, nocontour;
  Vec    *vsub, v;
  Draw   *draw;
  DF     df, df_local;
  DFVec  dfv1;
  Scalar *va, *vak;
  double *c0, *c1, h;
  char   filebase[1024], filename[1024], strnum[64];

  /* Form a local vector on processor 0 with the mapped ordering 
     Eventually we should work with sections of the vector, 1 at a time,
     instead of assembling the full parallel vector on 1 processor. */
  ierr = DFVecFormUniVec_MPIRegular_Private(dfv,&v); CHKERRQ(ierr);
  ierr = DFVecGetDFShell(dfv,&df); CHKERRQ(ierr);
  MPI_Comm_rank(df->comm,&rank);

  /* Fill up x and y coordinates */
  if (df->cset) {
    c0 = df->coord[0];
    c1 = df->coord[1];   
  } else {
    m = df->gnp[0]; n = df->gnp[1];
    c0 = (double *) PetscMalloc( (m+n)*sizeof(double) ); CHKPTRQ(c0);
    c1 = c0 + m;
    h = 1.0/(m-1);
    c0[0] = 0.0;
    for ( i=1; i<m; i++ ) c0[i] = c0[i-1] + h;
    h = 1.0/(n-1);
    c1[0] = 0.0;
    for ( i=1; i<n; i++ ) c1[i] = c1[i-1] + h;
  }

  /* Output file */
  ierr = OptionsHasName(PETSC_NULL,"-dfvec_vrml_nocontour",&nocontour);
  ierr = OptionsGetString(PETSC_NULL,"-dfvec_vrml_basefile",filebase,1000,&flg); CHKERRQ(ierr);
  if (!flg) PetscStrcpy(filebase,"dfvec");
  if (rank == 0) {
    ierr = DFShellGetLocalDFShell(df,&df_local);
    ierr = DFVecShellAssociate(df_local,v); CHKERRQ(ierr);
    dfv1 = (DFVec) v;
    ierr = DFVecGetComponentVectors(dfv1,&nc,&vsub); CHKERRQ(ierr);
    PLogObjectParents(dfv,nc,vsub);
    draw = (Draw *) PetscMalloc(nc*sizeof(Draw *));
    for (i=0; i<nc; i++) {
      ierr = VecGetArray(vsub[i],&va); CHKERRQ(ierr);
      vak  = va;
      sprintf(strnum,".%d.vrml",i);
      PetscStrcpy(filename,filebase);
      PetscStrcat(filename,strnum);
      ierr = DrawOpenVRML(v->comm,filename,df->label[i],&draw[i]); CHKERRQ(ierr);
      PLogObjectParent(df,draw[i]);
      for (k=0; k<df->gnp[2]; k++) {
        ierr = DrawBOP(draw[i]); CHKERRQ(ierr);
        if (nocontour) {
          ierr = DrawTensorSurface_VRML(draw[i],c0,df->gnp[0],
                   c1,df->gnp[1],vak,64); CHKERRQ(ierr);
        } else {
          ierr = DrawTensorSurfaceContour_VRML(draw[i],c0,df->gnp[0],
                   c1,df->gnp[1],vak,64); CHKERRQ(ierr);
        }
        ierr = DrawEOP(draw[i]); CHKERRQ(ierr);
        vak += df->gnp[0]*df->gnp[1];
      }
      ierr = VecRestoreArray(vsub[i],&va); CHKERRQ(ierr);
    }
    for (i=0; i<nc; i++) {ierr = DrawDestroy(draw[i]); CHKERRQ(ierr);}
    ierr = VecDestroyVecs(vsub,nc); CHKERRQ(ierr);
    ierr = VecDestroy(v); CHKERRQ(ierr);
    if (!df->cset) PetscFree(c0);
    PetscFree(draw);
  }
#endif
  PetscFunctionReturn(0);
}

extern int DFVecAssembleFullVector_MPIGeneral(Vec*,DFVec);

static struct _DFOps DFMPIRegular = 
  {DFVecRefineVector_MPIRegular,
   DFRefineCoordinates_SeqRegular,
   DFVecAssembleFullVector_MPIGeneral,
   DFVecDrawTensorContoursX_MPIRegular,
   DFVecDrawTensorContours_MPIRegular,
   DFVecDrawTensorSurfaceContoursVRML_MPIRegular,
   DFCopy_SeqRegular};

#undef __FUNC__  
#define __FUNC__ "DFSetMPIRegularOps_Private"
static int DFSetMPIRegularOps_Private(DF v)
{ 
  PetscMemcpy(v->ops,&DFMPIRegular,sizeof(DFMPIRegular));
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "DFShellCreateDA_Private"
/*
   DFShellCreateDA_Private - Creates a discrete function object from a vector
   associated with a distributed array (DA).  This routine is called internally
   by DACreate[1,2,3]d() and is not intended for users to employ directly.

   Input Parameters:
.  comm - MPI communicator
.  label - optional character string associated with each component
.  da - distributed array associated with the vector, formed
     by DACreate1d(), DACreate2d(), or DACreate3d()

   Output Parameter:
.  DF - grid info context

   Notes:
   For operations such as refinement and
   contour plotting, we currently support only vectors that are defined
   on regular grids with no mappings; we may extend this eventually.

   The grid point ordering for discrete functions with multiple degrees
   of freedom per node is ORDER_1, since this is used by the DAs.

   For more general use of discrete functions, use the routine
   DFShellCreate().

.seealso: DFShellCreate(), DFShellDestroy(), DFShellSetCoordinates()
*/
int DFShellCreateDA_Private(MPI_Comm comm,char **label,DA da,DF* df)
{
  DF       v, v_local;
  int      i, ierr, flag, xs, xm, ys, ym, zs, zm;
  char     *vl;
  MPI_Comm da_comm;

  PetscValidHeaderSpecific(da,DA_COOKIE);
  ierr = PetscObjectGetComm((PetscObject)da,&da_comm); CHKERRQ(ierr);
  ierr = MPI_Comm_compare(da_comm,comm,&flag);CHKERRQ(ierr);
  if (flag == MPI_UNEQUAL) {
    SETERRQ(PETSC_ERR_ARG_INCOMP,0,"Incompatible communicators for DF context and DA");
  }

  PetscHeaderCreate(v,_p_DF,struct _DFOps,DF_COOKIE,DF_MPIREG,comm,DFShellDestroy,0);
  PLogObjectCreate(v); 
  PLogObjectMemory(v,sizeof(struct _p_DF));
  ierr = DAGetInfo(da,&v->dim,PETSC_NULL,PETSC_NULL,PETSC_NULL,PETSC_NULL,
         PETSC_NULL,PETSC_NULL,&v->nc,PETSC_NULL,PETSC_NULL); CHKERRQ(ierr);

  v->da_user = da;
  v->da      = 0;
  v->order   = ORDER_1;
  v->cflag   = 0;
  if (label != PETSC_NULL) {
    v->label = label; v->ldefault = 0;
  } else {
    v->label = (char **) PetscMalloc(v->nc*sizeof(char*)); CHKPTRQ(v->label);
    for (i=0; i<v->nc; i++) {
      v->label[i] = vl = (char *) PetscMalloc(25*sizeof(char)); CHKPTRQ(vl);
      sprintf(vl,"Vector Component %d",i);
    }
    v->ldefault = 1;
    PLogObjectMemory(v,(v->nc*(sizeof(char*)+25*sizeof(char))));
  }
  v->coord[0]  = 0;  v->coord[1] = 0; v->coord[2] = 0;
  v->cset      = 0;
  v->rset      = 0;
  v->calloc    = 0;
  v->user      = 0;
  v->user_copy = 0;

  ierr = DAGetInfo(v->da_user,PETSC_NULL,&v->gnp[0],&v->gnp[1],&v->gnp[2],PETSC_NULL,
         PETSC_NULL,PETSC_NULL,PETSC_NULL,PETSC_NULL,PETSC_NULL); CHKERRQ(ierr);
  ierr = DAGetCorners(v->da_user,&xs,&ys,&zs,&xm,&ym,&zm); CHKERRQ(ierr);
  if (!xm) xm++; if (!ym) ym++; if (!zm) zm++;
  v->lnp[0] = zm; v->lnp[1] = ym; v->lnp[2] = zm;

  ierr = DFSetMPIRegularOps_Private(v); CHKERRQ(ierr);
  v->ops->view    = DFVecView_MPIRegular;
  ierr = DFShellCreate(PETSC_COMM_SELF,DF_SEQREG,v->dim,v->nc,v->order,label,
         v->lnp[0],v->lnp[1],v->lnp[2],&v_local); CHKERRQ(ierr);
  v->df_local = (void*)v_local;
  PLogObjectParent(v,v_local);
  *df = v;
  PetscFunctionReturn(0);
}





