#ifdef MPI_BUILD_PROFILING
#undef MPI_BUILD_PROFILING
#endif
#include <stdio.h>
#include <stdlib.h>
#include "mpi.h"
#include "mpe.h"

/* Enable memory tracing.  This requires MPICH's mpid/util/tr2.c codes */
#if defined(MPIR_MEMDEBUG)
#define malloc(a)    MPID_trmalloc((unsigned)(a),__LINE__,__FILE__)
#define free(a)      MPID_trfree(a,__LINE__,__FILE__)
#endif

/* 
   This is a large file and may exceed some compilers ability to handle.
   In that case, compile without optimization.
 */

/*
 * To give better control over the log file generated by these routines,
 * the user can switch on an off individual states as well as entire classes
 * of routines.  By default, only the communication routines, pack, and
 * unpack are logged.  This is a change from the previous implementation.
 *
 * MPI_Init checks for logging control options and environment variables,
 * and MPI_Pcontrol allows control over logging (allowing the user to
 * turn logging on and off).  Note that some routines are ALWAYS logged;
 * principly, these are the communicator constuction routines (needed to
 * avoid using the "context_id" which may not exist in some MPI
 * implementations).
 *
 * kind_mask is used ONLY when setting (or clearing) is_active by class.
 *
 * An additional feature is the RECV_IDLE state; this is the state between
 * when a recv is posted and when it is able to complete (basically the
 * time for an MPI_PROBE to succeed).
 *
 * Additional issues:
 * Must log all communicator creation events, even if logging turned off
 * (needed to match up communicator/contexts if a context_id not available).
 * We might choose to allow an MPIR_Comm_id(MPI_Comm) routine in
 * context/comm_util.c to simplify the implementation.
 *
 * We need begin-state and end-state routines, because we (a) might want
 * to use alternate logging routines (just link with a different logging
 * library to get runtime animation) and (b) some logging libraries might
 * want it that way (state-based instead of event-based).
 *
 * In request completion routines (e.g., wait, test), we need to shadow
 * the requests because a completed non-persistent request is freed and
 * the input value set to zero (a previous version of this code did not
 * do this).
 *
 * Note that some sends/recvs can be to/from MPI_PROC_NULL.  Must decide
 * whether no log event should be generated or not (can upshot/nupshot
 * handle MPI_PROC_NULL partners?  Should they?)
 * Previous code was uneven about that.
 *
 * Previous code was also uneven about placement of log_send/recv and
 * particularly on the size logged (some logged count, some size in bytes).
 * This still needs work.
 *
 * An alternate version of this could use macros (and require recompiling).
 */
typedef struct {
    int n_calls,      /* Number of times this state used */
        is_active,    /* Allows each state to be selectively switched off */
        id,           /* Id number of state */
        kind_mask;    /* Indicates kind of state (message, environment) */
    char *name;       /* Pointer to name */
    char *color;      /* Color (or B&W representation) */
} MPE_State;

/* Kind_mask values */
#define MPE_KIND_MSG 0x1
#define MPE_KIND_TOPO 0x2
#define MPE_KIND_COLL 0x4
#define MPE_KIND_DATATYPE 0x8
#define MPE_KIND_ENV 0x10
#define MPE_KIND_COMM_INFO 0x20
#define MPE_KIND_COMM 0x40
#define MPE_KIND_ATTR 0x80
#define MPE_KIND_GROUP 0x100
#define MPE_KIND_MSG_INIT 0x200
/* More as needed */

/* Number of MPI routines; increase to allow user extensions */
#define MPE_MAX_STATES 128
static MPE_State states[MPE_MAX_STATES];
/* Global trace control */
static int trace_on = 0;

#define MPE_ALLGATHER_ID 0
#define MPE_ALLGATHERV_ID 1
#define MPE_ALLREDUCE_ID 2
#define MPE_ALLTOALL_ID 3
#define MPE_ALLTOALLV_ID 4
#define MPE_BARRIER_ID 5
#define MPE_BCAST_ID 6
#define MPE_GATHER_ID 7
#define MPE_GATHERV_ID 8
#define MPE_OP_CREATE_ID 9
#define MPE_OP_FREE_ID 10
#define MPE_REDUCE_SCATTER_ID 11
#define MPE_REDUCE_ID 12
#define MPE_SCAN_ID 13
#define MPE_SCATTER_ID 14
#define MPE_SCATTERV_ID 15
#define MPE_ATTR_DELETE_ID 16
#define MPE_ATTR_GET_ID 17
#define MPE_ATTR_PUT_ID 18
#define MPE_COMM_COMPARE_ID 19
#define MPE_COMM_CREATE_ID 20
#define MPE_COMM_DUP_ID 21
#define MPE_COMM_FREE_ID 22
#define MPE_COMM_GROUP_ID 23
#define MPE_COMM_RANK_ID 24
#define MPE_COMM_REMOTE_GROUP_ID 25
#define MPE_COMM_REMOTE_SIZE_ID 26
#define MPE_COMM_SIZE_ID 27
#define MPE_COMM_SPLIT_ID 28
#define MPE_COMM_TEST_INTER_ID 29
#define MPE_GROUP_COMPARE_ID 30
#define MPE_GROUP_DIFFERENCE_ID 31
#define MPE_GROUP_EXCL_ID 32
#define MPE_GROUP_FREE_ID 33
#define MPE_GROUP_INCL_ID 34
#define MPE_GROUP_INTERSECTION_ID 35
#define MPE_GROUP_RANK_ID 36
#define MPE_GROUP_RANGE_EXCL_ID 37
#define MPE_GROUP_RANGE_INCL_ID 38
#define MPE_GROUP_SIZE_ID 39
#define MPE_GROUP_TRANSLATE_RANKS_ID 40
#define MPE_GROUP_UNION_ID 41
#define MPE_INTERCOMM_CREATE_ID 42
#define MPE_INTERCOMM_MERGE_ID 43
#define MPE_KEYVAL_CREATE_ID 44
#define MPE_KEYVAL_FREE_ID 45
#define MPE_ABORT_ID 46
#define MPE_ERROR_CLASS_ID 47
#define MPE_ERRHANDLER_CREATE_ID 48
#define MPE_ERRHANDLER_FREE_ID 49
#define MPE_ERRHANDLER_GET_ID 50
#define MPE_ERROR_STRING_ID 51
#define MPE_ERRHANDLER_SET_ID 52
#define MPE_GET_PROCESSOR_NAME_ID 53
#define MPE_INITIALIZED_ID 54
#define MPE_WTICK_ID 55
#define MPE_WTIME_ID 56
#define MPE_ADDRESS_ID 57
#define MPE_BSEND_ID 58
#define MPE_BSEND_INIT_ID 59
#define MPE_BUFFER_ATTACH_ID 60
#define MPE_BUFFER_DETACH_ID 61
#define MPE_CANCEL_ID 62
#define MPE_REQUEST_FREE_ID 63
#define MPE_RECV_INIT_ID 64
#define MPE_SEND_INIT_ID 65
#define MPE_GET_ELEMENTS_ID 66
#define MPE_GET_COUNT_ID 67
#define MPE_IBSEND_ID 68
#define MPE_IPROBE_ID 69
#define MPE_IRECV_ID 70
#define MPE_IRSEND_ID 71
#define MPE_ISEND_ID 72
#define MPE_ISSEND_ID 73
#define MPE_PACK_ID 74
#define MPE_PACK_SIZE_ID 75
#define MPE_PROBE_ID 76
#define MPE_RECV_ID 77
#define MPE_RSEND_ID 78
#define MPE_RSEND_INIT_ID 79
#define MPE_SEND_ID 80
#define MPE_SENDRECV_ID 81
#define MPE_SENDRECV_REPLACE_ID 82
#define MPE_SSEND_ID 83
#define MPE_SSEND_INIT_ID 84
#define MPE_START_ID 85
#define MPE_STARTALL_ID 86
#define MPE_TEST_ID 87
#define MPE_TESTALL_ID 88
#define MPE_TESTANY_ID 89
#define MPE_TEST_CANCELLED_ID 90
#define MPE_TESTSOME_ID 91
#define MPE_TYPE_COMMIT_ID 92
#define MPE_TYPE_CONTIGUOUS_ID 93
#define MPE_TYPE_EXTENT_ID 94
#define MPE_TYPE_FREE_ID 95
#define MPE_TYPE_HINDEXED_ID 96
#define MPE_TYPE_HVECTOR_ID 97
#define MPE_TYPE_INDEXED_ID 98
#define MPE_TYPE_LB_ID 99
#define MPE_TYPE_SIZE_ID 100
#define MPE_TYPE_STRUCT_ID 101
#define MPE_TYPE_UB_ID 102
#define MPE_TYPE_VECTOR_ID 103
#define MPE_UNPACK_ID 104
#define MPE_WAIT_ID 105
#define MPE_WAITALL_ID 106
#define MPE_WAITANY_ID 107
#define MPE_WAITSOME_ID 108
#define MPE_CART_COORDS_ID 109
#define MPE_CART_CREATE_ID 110
#define MPE_CART_GET_ID 111
#define MPE_CART_MAP_ID 112
#define MPE_CART_SHIFT_ID 113
#define MPE_CARTDIM_GET_ID 114
#define MPE_DIMS_CREATE_ID 115
#define MPE_GRAPH_CREATE_ID 116
#define MPE_GRAPH_GET_ID 117
#define MPE_GRAPH_MAP_ID 118
#define MPE_GRAPH_NEIGHBORS_ID 119
#define MPE_GRAPH_NEIGHBORS_COUNT_ID 120
#define MPE_GRAPHDIMS_GET_ID 121
#define MPE_TOPO_TEST_ID 122
#define MPE_RECV_IDLE_ID 123
#define MPE_CART_RANK_ID 124
#define MPE_CART_SUB_ID 125

#include "requests.h"

static request_list *requests_head_0, *requests_tail_0, *requests_avail_0=0;
static int procid_0;
static char logFileName_0[256];

/* This is used for the multiple-completion test/wait functions */
#define MPE_MAX_REQUESTS 16
static MPI_Request req[MPE_MAX_REQUESTS];

/* Function prototypes */
void MPE_Add_send_req ANSI_ARGS(( int, MPI_Datatype, int, int, 
				  MPI_Request, int ));
void MPE_Add_recv_req ANSI_ARGS(( int, MPI_Datatype, int, int, 
				  MPI_Request, int ));
void MPE_Cancel_req ANSI_ARGS(( MPI_Request ));
void MPE_Remove_req ANSI_ARGS(( MPI_Request ));
void MPE_Start_req ANSI_ARGS(( MPI_Request, MPE_State * ));
void MPE_ProcessWaitTest ANSI_ARGS(( MPI_Request, MPI_Status *, char *,
				     MPE_State * ));

/*
   Temporary MPE log definitions (eventually will replace with more
   permanent changes)
   Note that these include a communicator as well as the state (pointer
   to predefined state structure).  Use MPI_COMM_NULL for no communicator
 */
#define MPE_Log_state_begin( comm, state ) \
    MPE_Log_event( state->id, state->n_calls, (char *)0 )
#define MPE_Log_state_end( comm, state ) \
    MPE_Log_event( state->id + 1, state->n_calls, (char *)0 )

/* To use these, declare
     register MPE_State *state;
   and call around routine */
#define MPE_LOG_STATE_BEGIN(name,comm) \
  if (trace_on) {\
      state = &states[name];\
      if (state->is_active) {\
      	  state->n_calls++;\
          MPE_Log_state_begin( comm, state );\
      }\
  }
#define MPE_LOG_STATE_END(comm) \
  if (trace_on && state->is_active) {\
      	  state->n_calls++;\
          MPE_Log_state_end( comm, state );\
      }
#define MPE_LOG_DO(call) \
    if (trace_on && state->is_active) { call ; }

/* Service routines for managing requests .... */
/* If there are large numbers of requests, we should probably use a better
   search structure, such as a hash table or tree
 */
void MPE_Add_send_req( count, datatype, dest, tag, request, is_persistent )
int count, dest, tag, is_persistent;
MPI_Datatype datatype;
MPI_Request request;
{
  request_list *newrq;
  int typesize;

  rq_alloc(requests_avail_0,newrq);
  if (newrq) {
      PMPI_Type_size( datatype, &typesize );
      newrq->request	   = request;
      newrq->status	   = RQ_SEND;
      newrq->size	   = count * typesize;
      newrq->tag	   = tag;
      newrq->otherParty	   = dest;
      newrq->next	   = 0;
      newrq->is_persistent = is_persistent;
      rq_add( requests_head_0, requests_tail_0, newrq );
    }
}

void MPE_Add_recv_req( count, datatype, source, tag, request, is_persistent )
int count, source, tag, is_persistent;
MPI_Datatype datatype;
MPI_Request request;
{
  request_list *newrq;

  /* We could pre-allocate request_list members, or allocate in
     blocks.  Do this is we see this is a bottleneck */
  rq_alloc( requests_avail_0, newrq );
  if (newrq) {
      newrq->request	   = request;
      newrq->status	   = RQ_RECV;
      newrq->next	   = 0;
      newrq->is_persistent = is_persistent;
      rq_add( requests_head_0, requests_tail_0, newrq );
    }
}

void  MPE_Cancel_req( request )
MPI_Request request;
{
    request_list *rq;
    rq_find( requests_head_0, request, rq );
    if (rq) rq->status |= RQ_CANCEL;
}

void  MPE_Remove_req( request )
MPI_Request request;
{
  rq_remove( requests_head_0, requests_tail_0, requests_avail_0, request );
}

/* Persistent sends and receives are handled with this routine (called by
   start or startall) */
void MPE_Start_req( request, state )
MPI_Request request;
MPE_State   *state;
{
  request_list *rq;

  /* look for request */
  rq = requests_head_0;
  while (rq && (rq->request != request)) {
    rq   = rq->next;
  }

  if (!rq) {
#ifdef PRINT_PROBLEMS
    fprintf( stderr, "Request not found in '%s'.\n", note );
#endif
    return;		/* request not found */
  }

  if ((rq->status & RQ_SEND) && rq->otherParty != MPI_PROC_NULL) {
      MPE_LOG_DO(MPE_Log_send( rq->otherParty, rq->tag, rq->size ));
  }

}
   
void MPE_ProcessWaitTest ( request, status, note, state )
MPI_Request request;
MPI_Status  *status;
char        *note;
MPE_State   *state;
{
  request_list *rq, *last;
  int flag, size;

  /* look for request */
  rq = requests_head_0;
  last = 0;
  while (rq && (rq->request != request)) {
    last = rq;
    rq   = rq->next;
  }

  if (!rq) {
#ifdef PRINT_PROBLEMS
    fprintf( stderr, "Request not found in '%s'.\n", note );
#endif
    return;		/* request not found */
  }

  if (status->MPI_TAG != MPI_ANY_TAG || (rq->status & RQ_SEND) ) {
    /* if the request was not invalid */

    if (rq->status & RQ_CANCEL) {
      PMPI_Test_cancelled( status, &flag );
      if (flag) return;		/* the request has been cancelled */
    }

    /* Receives conclude at the END (wait/test); sends start at the
       beginning */    
    if ((rq->status & RQ_RECV) && (status->MPI_SOURCE != MPI_PROC_NULL)) {
      PMPI_Get_count( status, MPI_BYTE, &size );
      MPE_LOG_DO(MPE_Log_receive( status->MPI_SOURCE, status->MPI_TAG, size ));
    }
  }

  /* Since we already have found the request, removing it is relatively
     easy */
  if (!rq->is_persistent) {
      rq_remove_at( requests_head_0, requests_tail_0, requests_avail_0, 
		    rq, last );
  }
}


/*
 * Here begins the individual routines.  We may eventually want to
 * break them up, at least by class (no need to load the MPI_CART/GRAPH
 * routines if the application doesn't use them).
 */


int   MPI_Allgather( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, comm )
void * sendbuf;
int sendcount;
MPI_Datatype sendtype;
void * recvbuf;
int recvcount;
MPI_Datatype recvtype;
MPI_Comm comm;
{
  int       returnVal;
  register MPE_State *state;

/*
    MPI_Allgather - prototyping replacement for MPI_Allgather
    Log the beginning and ending of the time spent in MPI_Allgather calls.
*/
  MPE_LOG_STATE_BEGIN(MPE_ALLGATHER_ID,comm);
  
  returnVal = PMPI_Allgather( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Allgatherv( sendbuf, sendcount, sendtype, recvbuf, recvcounts, displs, recvtype, comm )
void * sendbuf;
int sendcount;
MPI_Datatype sendtype;
void * recvbuf;
int * recvcounts;
int * displs;
MPI_Datatype recvtype;
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;
/*
    MPI_Allgatherv - prototyping replacement for MPI_Allgatherv
    Log the beginning and ending of the time spent in MPI_Allgatherv calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ALLGATHERV_ID, comm );
  
  returnVal = PMPI_Allgatherv( sendbuf, sendcount, sendtype, recvbuf, recvcounts, displs, recvtype, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Allreduce( sendbuf, recvbuf, count, datatype, op, comm )
void * sendbuf;
void * recvbuf;
int count;
MPI_Datatype datatype;
MPI_Op op;
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Allreduce - prototyping replacement for MPI_Allreduce
    Log the beginning and ending of the time spent in MPI_Allreduce calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ALLREDUCE_ID,comm);
  
  returnVal = PMPI_Allreduce( sendbuf, recvbuf, count, datatype, op, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Alltoall( sendbuf, sendcount, sendtype, recvbuf, recvcnt, recvtype, comm )
void * sendbuf;
int sendcount;
MPI_Datatype sendtype;
void * recvbuf;
int recvcnt;
MPI_Datatype recvtype;
MPI_Comm comm;
{
  int  returnVal;
  register MPE_State *state;
  
/*
    MPI_Alltoall - prototyping replacement for MPI_Alltoall
    Log the beginning and ending of the time spent in MPI_Alltoall calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ALLTOALL_ID,comm);
  
  returnVal = PMPI_Alltoall( sendbuf, sendcount, sendtype, recvbuf, recvcnt, recvtype, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Alltoallv( sendbuf, sendcnts, sdispls, sendtype, recvbuf, recvcnts, rdispls, recvtype, comm )
void * sendbuf;
int * sendcnts;
int * sdispls;
MPI_Datatype sendtype;
void * recvbuf;
int * recvcnts;
int * rdispls;
MPI_Datatype recvtype;
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;
/*
    MPI_Alltoallv - prototyping replacement for MPI_Alltoallv
    Log the beginning and ending of the time spent in MPI_Alltoallv calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ALLTOALLV_ID,comm);
  
  returnVal = PMPI_Alltoallv( sendbuf, sendcnts, sdispls, sendtype, recvbuf, recvcnts, rdispls, recvtype, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Barrier( comm )
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;
/*
    MPI_Barrier - prototyping replacement for MPI_Barrier
    Log the beginning and ending of the time spent in MPI_Barrier calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_BARRIER_ID,comm);
  
  returnVal = PMPI_Barrier( comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int MPI_Bcast( buffer, count, datatype, root, comm )
void * buffer;
int count;
MPI_Datatype datatype;
int root;
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;
/*
    MPI_Bcast - prototyping replacement for MPI_Bcast
    Log the beginning and ending of the time spent in MPI_Bcast calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_BCAST_ID,comm);
  
  returnVal = PMPI_Bcast( buffer, count, datatype, root, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int MPI_Gather( sendbuf, sendcnt, sendtype, recvbuf, recvcount, recvtype, root, comm )
void * sendbuf;
int sendcnt;
MPI_Datatype sendtype;
void * recvbuf;
int recvcount;
MPI_Datatype recvtype;
int root;
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;
/*
    MPI_Gather - prototyping replacement for MPI_Gather
    Log the beginning and ending of the time spent in MPI_Gather calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GATHER_ID,comm);
  
  returnVal = PMPI_Gather( sendbuf, sendcnt, sendtype, recvbuf, recvcount, recvtype, root, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int MPI_Gatherv( sendbuf, sendcnt, sendtype, recvbuf, recvcnts, displs, recvtype, root, comm )
void * sendbuf;
int sendcnt;
MPI_Datatype sendtype;
void * recvbuf;
int * recvcnts;
int * displs;
MPI_Datatype recvtype;
int root;
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;
/*
    MPI_Gatherv - prototyping replacement for MPI_Gatherv
    Log the beginning and ending of the time spent in MPI_Gatherv calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GATHERV_ID,comm);
  
  returnVal = PMPI_Gatherv( sendbuf, sendcnt, sendtype, recvbuf, recvcnts, displs, recvtype, root, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Op_create( function, commute, op )
MPI_User_function * function;
int commute;
MPI_Op * op;
{
  int  returnVal;
  register MPE_State *state;
/*
    MPI_Op_create - prototyping replacement for MPI_Op_create
    Log the beginning and ending of the time spent in MPI_Op_create calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_OP_CREATE_ID,(MPI_Comm *)0);
  
  returnVal = PMPI_Op_create( function, commute, op );

  MPE_LOG_STATE_END((MPI_Comm*)0);

  return returnVal;
}

int  MPI_Op_free( op )
MPI_Op * op;
{
  int  returnVal;
  register MPE_State *state;
  
/*
    MPI_Op_free - prototyping replacement for MPI_Op_free
    Log the beginning and ending of the time spent in MPI_Op_free calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_OP_FREE_ID,(MPI_Comm*)0);
  
  returnVal = PMPI_Op_free( op );

  MPE_LOG_STATE_END((MPI_Comm*)0);

  return returnVal;
}

int   MPI_Reduce_scatter( sendbuf, recvbuf, recvcnts, datatype, op, comm )
void * sendbuf;
void * recvbuf;
int * recvcnts;
MPI_Datatype datatype;
MPI_Op op;
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;
/*
    MPI_Reduce_scatter - prototyping replacement for MPI_Reduce_scatter
    Log the beginning and ending of the time spent in MPI_Reduce_scatter calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_REDUCE_SCATTER_ID,comm);
  
  returnVal = PMPI_Reduce_scatter( sendbuf, recvbuf, recvcnts, datatype, op, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Reduce( sendbuf, recvbuf, count, datatype, op, root, comm )
void * sendbuf;
void * recvbuf;
int count;
MPI_Datatype datatype;
MPI_Op op;
int root;
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;
/*
    MPI_Reduce - prototyping replacement for MPI_Reduce
    Log the beginning and ending of the time spent in MPI_Reduce calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_REDUCE_ID,comm);
  
  returnVal = PMPI_Reduce( sendbuf, recvbuf, count, datatype, op, root, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Scan( sendbuf, recvbuf, count, datatype, op, comm )
void * sendbuf;
void * recvbuf;
int count;
MPI_Datatype datatype;
MPI_Op op;
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;
  
/*
    MPI_Scan - prototyping replacement for MPI_Scan
    Log the beginning and ending of the time spent in MPI_Scan calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_SCAN_ID,comm);
  
  returnVal = PMPI_Scan( sendbuf, recvbuf, count, datatype, op, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Scatter( sendbuf, sendcnt, sendtype, recvbuf, recvcnt, recvtype, root, comm )
void * sendbuf;
int sendcnt;
MPI_Datatype sendtype;
void * recvbuf;
int recvcnt;
MPI_Datatype recvtype;
int root;
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;
  
/*
    MPI_Scatter - prototyping replacement for MPI_Scatter
    Log the beginning and ending of the time spent in MPI_Scatter calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_SCATTER_ID,comm);
  
  returnVal = PMPI_Scatter( sendbuf, sendcnt, sendtype, recvbuf, recvcnt, recvtype, root, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Scatterv( sendbuf, sendcnts, displs, sendtype,
                    recvbuf, recvcnt, recvtype, root, comm )
void * sendbuf;
int * sendcnts;
int * displs;
MPI_Datatype sendtype;
void * recvbuf;
int recvcnt;
MPI_Datatype recvtype;
int root;
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;
  
/*
    MPI_Scatterv - prototyping replacement for MPI_Scatterv
    Log the beginning and ending of the time spent in MPI_Scatterv calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_SCATTERV_ID,comm);
  
  returnVal = PMPI_Scatterv( sendbuf, sendcnts, displs, sendtype, recvbuf, recvcnt, recvtype, root, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Attr_delete( comm, keyval )
MPI_Comm comm;
int keyval;
{
  int   returnVal;
  register MPE_State *state;
  
/*
    MPI_Attr_delete - prototyping replacement for MPI_Attr_delete
    Log the beginning and ending of the time spent in MPI_Attr_delete calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ATTR_DELETE_ID,comm);
  
  returnVal = PMPI_Attr_delete( comm, keyval );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Attr_get( comm, keyval, attr_value, flag )
MPI_Comm comm;
int keyval;
void * attr_value;
int * flag;
{
  int   returnVal;
  register MPE_State *state;
  
/*
    MPI_Attr_get - prototyping replacement for MPI_Attr_get
    Log the beginning and ending of the time spent in MPI_Attr_get calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ATTR_GET_ID,comm);
  
  returnVal = PMPI_Attr_get( comm, keyval, attr_value, flag );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Attr_put( comm, keyval, attr_value )
MPI_Comm comm;
int keyval;
void * attr_value;
{
  int   returnVal;
  register MPE_State *state;
/*
    MPI_Attr_put - prototyping replacement for MPI_Attr_put
    Log the beginning and ending of the time spent in MPI_Attr_put calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ATTR_PUT_ID,comm);
  
  returnVal = PMPI_Attr_put( comm, keyval, attr_value );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Comm_compare( comm1, comm2, result )
MPI_Comm comm1;
MPI_Comm comm2;
int * result;
{
  int   returnVal;
  register MPE_State *state;
/*
    MPI_Comm_compare - prototyping replacement for MPI_Comm_compare
    Log the beginning and ending of the time spent in MPI_Comm_compare calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_COMM_COMPARE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Comm_compare( comm1, comm2, result );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Comm_create( comm, group, comm_out )
MPI_Comm comm;
MPI_Group group;
MPI_Comm * comm_out;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Comm_create - prototyping replacement for MPI_Comm_create
    Log the beginning and ending of the time spent in MPI_Comm_create calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_COMM_CREATE_ID,comm);
  
  returnVal = PMPI_Comm_create( comm, group, comm_out );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Comm_dup( comm, comm_out )
MPI_Comm comm;
MPI_Comm * comm_out;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Comm_dup - prototyping replacement for MPI_Comm_dup
    Log the beginning and ending of the time spent in MPI_Comm_dup calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_COMM_DUP_ID,comm);
  
  returnVal = PMPI_Comm_dup( comm, comm_out );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Comm_free( comm )
MPI_Comm * comm;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Comm_free - prototyping replacement for MPI_Comm_free
    Log the beginning and ending of the time spent in MPI_Comm_free calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_COMM_FREE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Comm_free( comm );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Comm_group( comm, group )
MPI_Comm comm;
MPI_Group * group;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Comm_group - prototyping replacement for MPI_Comm_group
    Log the beginning and ending of the time spent in MPI_Comm_group calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_COMM_GROUP_ID,comm);
  
  returnVal = PMPI_Comm_group( comm, group );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Comm_rank( comm, rank )
MPI_Comm comm;
int * rank;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Comm_rank - prototyping replacement for MPI_Comm_rank
    Log the beginning and ending of the time spent in MPI_Comm_rank calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_COMM_RANK_ID,comm);

  returnVal = PMPI_Comm_rank( comm, rank );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Comm_remote_group( comm, group )
MPI_Comm comm;
MPI_Group * group;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Comm_remote_group - prototyping replacement for MPI_Comm_remote_group
    Log the beginning and ending of the time spent in MPI_Comm_remote_group calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_COMM_REMOTE_GROUP_ID,comm);
  
  returnVal = PMPI_Comm_remote_group( comm, group );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Comm_remote_size( comm, size )
MPI_Comm comm;
int * size;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Comm_remote_size - prototyping replacement for MPI_Comm_remote_size
    Log the beginning and ending of the time spent in MPI_Comm_remote_size calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_COMM_REMOTE_SIZE_ID,comm);
  
  returnVal = PMPI_Comm_remote_size( comm, size );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Comm_size( comm, size )
MPI_Comm comm;
int * size;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Comm_size - prototyping replacement for MPI_Comm_size
    Log the beginning and ending of the time spent in MPI_Comm_size calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_COMM_SIZE_ID,comm);
  
  returnVal = PMPI_Comm_size( comm, size );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Comm_split( comm, color, key, comm_out )
MPI_Comm comm;
int color;
int key;
MPI_Comm * comm_out;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Comm_split - prototyping replacement for MPI_Comm_split
    Log the beginning and ending of the time spent in MPI_Comm_split calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_COMM_SPLIT_ID,comm);
  
  returnVal = PMPI_Comm_split( comm, color, key, comm_out );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Comm_test_inter( comm, flag )
MPI_Comm comm;
int * flag;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Comm_test_inter - prototyping replacement for MPI_Comm_test_inter
    Log the beginning and ending of the time spent in MPI_Comm_test_inter calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_COMM_TEST_INTER_ID,comm);
  
  returnVal = PMPI_Comm_test_inter( comm, flag );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Group_compare( group1, group2, result )
MPI_Group group1;
MPI_Group group2;
int * result;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Group_compare - prototyping replacement for MPI_Group_compare
    Log the beginning and ending of the time spent in MPI_Group_compare calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GROUP_COMPARE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Group_compare( group1, group2, result );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Group_difference( group1, group2, group_out )
MPI_Group group1;
MPI_Group group2;
MPI_Group * group_out;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Group_difference - prototyping replacement for MPI_Group_difference
    Log the beginning and ending of the time spent in MPI_Group_difference calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GROUP_DIFFERENCE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Group_difference( group1, group2, group_out );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Group_excl( group, n, ranks, newgroup )
MPI_Group group;
int n;
int * ranks;
MPI_Group * newgroup;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Group_excl - prototyping replacement for MPI_Group_excl
    Log the beginning and ending of the time spent in MPI_Group_excl calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GROUP_EXCL_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Group_excl( group, n, ranks, newgroup );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Group_free( group )
MPI_Group * group;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Group_free - prototyping replacement for MPI_Group_free
    Log the beginning and ending of the time spent in MPI_Group_free calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GROUP_FREE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Group_free( group );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Group_incl( group, n, ranks, group_out )
MPI_Group group;
int n;
int * ranks;
MPI_Group * group_out;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Group_incl - prototyping replacement for MPI_Group_incl
    Log the beginning and ending of the time spent in MPI_Group_incl calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GROUP_INCL_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Group_incl( group, n, ranks, group_out );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Group_intersection( group1, group2, group_out )
MPI_Group group1;
MPI_Group group2;
MPI_Group * group_out;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Group_intersection - prototyping replacement for MPI_Group_intersection
    Log the beginning and ending of the time spent in MPI_Group_intersection calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GROUP_INTERSECTION_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Group_intersection( group1, group2, group_out );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Group_rank( group, rank )
MPI_Group group;
int * rank;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Group_rank - prototyping replacement for MPI_Group_rank
    Log the beginning and ending of the time spent in MPI_Group_rank calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GROUP_RANK_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Group_rank( group, rank );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Group_range_excl( group, n, ranges, newgroup )
MPI_Group group;
int n;
int ranges[][3];
MPI_Group * newgroup;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Group_range_excl - prototyping replacement for MPI_Group_range_excl
    Log the beginning and ending of the time spent in MPI_Group_range_excl calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GROUP_RANGE_EXCL_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Group_range_excl( group, n, ranges, newgroup );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Group_range_incl( group, n, ranges, newgroup )
MPI_Group group;
int n;
int ranges[][3];
MPI_Group * newgroup;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Group_range_incl - prototyping replacement for MPI_Group_range_incl
    Log the beginning and ending of the time spent in MPI_Group_range_incl calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GROUP_RANGE_INCL_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Group_range_incl( group, n, ranges, newgroup );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Group_size( group, size )
MPI_Group group;
int * size;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Group_size - prototyping replacement for MPI_Group_size
    Log the beginning and ending of the time spent in MPI_Group_size calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GROUP_SIZE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Group_size( group, size );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Group_translate_ranks( group_a, n, ranks_a, group_b, ranks_b )
MPI_Group group_a;
int n;
int * ranks_a;
MPI_Group group_b;
int * ranks_b;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Group_translate_ranks - prototyping replacement for MPI_Group_translate_ranks
    Log the beginning and ending of the time spent in MPI_Group_translate_ranks calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GROUP_TRANSLATE_RANKS_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Group_translate_ranks( group_a, n, ranks_a, group_b, ranks_b );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Group_union( group1, group2, group_out )
MPI_Group group1;
MPI_Group group2;
MPI_Group * group_out;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Group_union - prototyping replacement for MPI_Group_union
    Log the beginning and ending of the time spent in MPI_Group_union calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GROUP_UNION_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Group_union( group1, group2, group_out );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Intercomm_create( local_comm, local_leader, peer_comm, remote_leader, tag, comm_out )
MPI_Comm local_comm;
int local_leader;
MPI_Comm peer_comm;
int remote_leader;
int tag;
MPI_Comm * comm_out;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Intercomm_create - prototyping replacement for MPI_Intercomm_create
    Log the beginning and ending of the time spent in MPI_Intercomm_create calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_INTERCOMM_CREATE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Intercomm_create( local_comm, local_leader, peer_comm, remote_leader, tag, comm_out );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Intercomm_merge( comm, high, comm_out )
MPI_Comm comm;
int high;
MPI_Comm * comm_out;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Intercomm_merge - prototyping replacement for MPI_Intercomm_merge
    Log the beginning and ending of the time spent in MPI_Intercomm_merge calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_INTERCOMM_MERGE_ID,comm);
  
  returnVal = PMPI_Intercomm_merge( comm, high, comm_out );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Keyval_create( copy_fn, delete_fn, keyval, extra_state )
MPI_Copy_function * copy_fn;
MPI_Delete_function * delete_fn;
int * keyval;
void * extra_state;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Keyval_create - prototyping replacement for MPI_Keyval_create
    Log the beginning and ending of the time spent in MPI_Keyval_create calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_KEYVAL_CREATE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Keyval_create( copy_fn, delete_fn, keyval, extra_state );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Keyval_free( keyval )
int * keyval;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Keyval_free - prototyping replacement for MPI_Keyval_free
    Log the beginning and ending of the time spent in MPI_Keyval_free calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_KEYVAL_FREE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Keyval_free( keyval );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Abort( comm, errorcode )
MPI_Comm comm;
int errorcode;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Abort - prototyping replacement for MPI_Abort
    Log the beginning and ending of the time spent in MPI_Abort calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ABORT_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Abort( comm, errorcode );

  /* Pretty implausible... */
  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Error_class( errorcode, errorclass )
int errorcode;
int * errorclass;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Error_class - prototyping replacement for MPI_Error_class
    Log the beginning and ending of the time spent in MPI_Error_class calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ERROR_CLASS_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Error_class( errorcode, errorclass );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Errhandler_create( function, errhandler )
MPI_Handler_function * function;
MPI_Errhandler * errhandler;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Errhandler_create - prototyping replacement for MPI_Errhandler_create
    Log the beginning and ending of the time spent in MPI_Errhandler_create calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ERRHANDLER_CREATE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Errhandler_create( function, errhandler );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Errhandler_free( errhandler )
MPI_Errhandler * errhandler;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Errhandler_free - prototyping replacement for MPI_Errhandler_free
    Log the beginning and ending of the time spent in MPI_Errhandler_free calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ERRHANDLER_FREE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Errhandler_free( errhandler );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Errhandler_get( comm, errhandler )
MPI_Comm comm;
MPI_Errhandler * errhandler;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Errhandler_get - prototyping replacement for MPI_Errhandler_get
    Log the beginning and ending of the time spent in MPI_Errhandler_get calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ERRHANDLER_GET_ID,comm);
  
  returnVal = PMPI_Errhandler_get( comm, errhandler );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Error_string( errorcode, string, resultlen )
int errorcode;
char * string;
int * resultlen;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Error_string - prototyping replacement for MPI_Error_string
    Log the beginning and ending of the time spent in MPI_Error_string calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ERROR_STRING_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Error_string( errorcode, string, resultlen );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Errhandler_set( comm, errhandler )
MPI_Comm comm;
MPI_Errhandler errhandler;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Errhandler_set - prototyping replacement for MPI_Errhandler_set
    Log the beginning and ending of the time spent in MPI_Errhandler_set calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ERRHANDLER_SET_ID,comm);
  
  returnVal = PMPI_Errhandler_set( comm, errhandler );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Finalize(  )
{
    int  returnVal, i;
    int  cnt[MPE_MAX_STATES], totcnt[MPE_MAX_STATES];
    
/*
    MPI_Finalize - prototyping replacement for MPI_Finalize
*/
  /* First, get the total number of calls by any processor */
    for (i=0; i<MPE_MAX_STATES; i++) 
  	cnt[i] = states[i].n_calls;
    PMPI_Reduce( cnt, totcnt, MPE_MAX_STATES, MPI_INT, MPI_SUM, 
		 0, MPI_COMM_WORLD );
    
  if (procid_0 == 0) {
    fprintf( stderr, "Writing logfile.\n");
    for (i=0; i<MPE_MAX_STATES; i++) {
        if (totcnt[i] > 0) {
  	    MPE_Describe_state( states[i].id, states[i].id + 1, 
				states[i].name, states[i].color );
  	}
    }
  }
  MPE_Finish_log( logFileName_0 );
  if (procid_0 == 0)
      fprintf( stderr, "Finished writing logfile.\n");

  /* Recover all of the allocated requests */
  rq_end( requests_avail_0 );

  returnVal = PMPI_Finalize(  );

  return returnVal;
}

int  MPI_Get_processor_name( name, resultlen )
char * name;
int * resultlen;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Get_processor_name - prototyping replacement for MPI_Get_processor_name
    Log the beginning and ending of the time spent in MPI_Get_processor_name calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GET_PROCESSOR_NAME_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Get_processor_name( name, resultlen );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

/*
 * Replacement for MPI_Init.  Initializes logging and sets up basic
 * state definitions, including default color/pattern values
 */
int  MPI_Init( argc, argv )
int * argc;
char *** argv;
{
  int  returnVal;
  int stateid, i;
  MPE_State *state;
  int allow_mask;

  returnVal = PMPI_Init( argc, argv );


  MPE_Init_log();
  PMPI_Comm_rank( MPI_COMM_WORLD, &procid_0 );
  stateid=1;
  /* Initialize all states */
  for (i=0; i<MPE_MAX_STATES; i++) {
  	states[i].n_calls   = 0;
  	states[i].id	    = stateid; stateid += 2;
  	states[i].is_active = 0;
  	states[i].name	    = 0;
  	states[i].kind_mask = 0;
  	states[i].color	    = "white:vlines";
  }

  /* By default, log only message-passing (pt-to-pt and collective) */
  allow_mask = MPE_KIND_MSG | MPE_KIND_COLL;
  /* Should check environment and command-line for changes to allow_mask */
  
  /* We COULD read these definitions from a file, but accessing the file
     in PARALLEL can be a problem and even if one process accessed it and
     broadcast, we'd still have to find the file.  Is this a problem?
     (We have to WRITE the file, after all).

     We only need to load the name and kind_mask.  is_active is derived
     from kind_mask and allowed mask.
   */
  state = &states[MPE_ALLGATHER_ID];
  state->kind_mask = MPE_KIND_COLL;
  state->name = "ALLGATHER";

  state = &states[MPE_ALLGATHERV_ID];
  state->kind_mask = MPE_KIND_COLL;
  state->name = "ALLGATHERV";

  state = &states[MPE_ALLREDUCE_ID];
  state->kind_mask = MPE_KIND_COLL;
  state->name = "ALLREDUCE";
  state->color = "purple:vlines3";

  state = &states[MPE_ALLTOALL_ID];
  state->kind_mask = MPE_KIND_COLL;
  state->name = "ALLTOALL";

  state = &states[MPE_ALLTOALLV_ID];
  state->kind_mask = MPE_KIND_COLL;
  state->name = "ALLTOALLV";

  state = &states[MPE_BARRIER_ID];
  state->kind_mask = MPE_KIND_COLL;
  state->name = "BARRIER";
  state->color = "yellow:dimple3";

  state = &states[MPE_BCAST_ID];
  state->kind_mask = MPE_KIND_COLL;
  state->name = "BCAST";
  state->color = "cyan:boxes";

  state = &states[MPE_GATHER_ID];
  state->kind_mask = MPE_KIND_COLL;
  state->name = "GATHER";

  state = &states[MPE_OP_CREATE_ID];
  state->kind_mask = MPE_KIND_ENV;
  state->name = "OP_CREATE";

  state = &states[MPE_OP_FREE_ID];
  state->kind_mask = MPE_KIND_ENV;
  state->name = "OP_FREE";

  state = &states[MPE_REDUCE_SCATTER_ID];
  state->kind_mask = MPE_KIND_COLL;
  state->name = "REDUCE_SCATTER";

  state = &states[MPE_REDUCE_ID];
  state->kind_mask = MPE_KIND_COLL;
  state->name = "REDUCE";
  state->color = "purple:2x2";

  state = &states[MPE_SCAN_ID];
  state->kind_mask = MPE_KIND_COLL;
  state->name = "SCAN";

  state = &states[MPE_SCATTER_ID];
  state->kind_mask = MPE_KIND_COLL;
  state->name = "SCATTER";

  state = &states[MPE_SCATTERV_ID];
  state->kind_mask = MPE_KIND_COLL;
  state->name = "SCATTERV";

  state = &states[MPE_ATTR_DELETE_ID];
  state->kind_mask = MPE_KIND_ATTR;
  state->name = "ATTR_DELETE";

  state = &states[MPE_ATTR_GET_ID];
  state->kind_mask = MPE_KIND_ATTR;
  state->name = "ATTR_GET";

  state = &states[MPE_ATTR_PUT_ID];
  state->kind_mask = MPE_KIND_ATTR;
  state->name = "ATTR_PUT";

  state = &states[MPE_COMM_COMPARE_ID];
  state->kind_mask = MPE_KIND_COMM_INFO;
  state->name = "COMM_COMPARE";

  state = &states[MPE_COMM_CREATE_ID];
  state->kind_mask = MPE_KIND_COMM;
  state->name = "COMM_CREATE";

  state = &states[MPE_COMM_DUP_ID];
  state->kind_mask = MPE_KIND_COMM;
  state->name = "COMM_DUP";

  state = &states[MPE_COMM_FREE_ID];
  state->kind_mask = MPE_KIND_COMM;
  state->name = "COMM_FREE";

  state = &states[MPE_COMM_GROUP_ID];
  state->kind_mask = MPE_KIND_COMM_INFO;
  state->name = "COMM_GROUP";

  state = &states[MPE_COMM_RANK_ID];
  state->kind_mask = MPE_KIND_COMM_INFO;
  state->name = "COMM_RANK";

  state = &states[MPE_COMM_REMOTE_GROUP_ID];
  state->kind_mask = MPE_KIND_COMM_INFO;
  state->name = "COMM_REMOTE_GROUP";

  state = &states[MPE_COMM_REMOTE_SIZE_ID];
  state->kind_mask = MPE_KIND_COMM_INFO;
  state->name = "COMM_REMOTE_SIZE";

  state = &states[MPE_COMM_SIZE_ID];
  state->kind_mask = MPE_KIND_COMM_INFO;
  state->name = "COMM_SIZE";

  state = &states[MPE_COMM_SPLIT_ID];
  state->kind_mask = MPE_KIND_COMM;
  state->name = "COMM_SPLIT";

  state = &states[MPE_COMM_TEST_INTER_ID];
  state->kind_mask = MPE_KIND_COMM_INFO;
  state->name = "COMM_TEST_INTER";

  state = &states[MPE_GROUP_COMPARE_ID];
  state->kind_mask = MPE_KIND_GROUP;
  state->name = "GROUP_COMPARE";

  state = &states[MPE_GROUP_DIFFERENCE_ID];  state->kind_mask = MPE_KIND_GROUP;
  state->name = "GROUP_DIFFERENCE";

  state = &states[MPE_GROUP_EXCL_ID];
  state->kind_mask = MPE_KIND_GROUP;
  state->name = "GROUP_EXCL";

  state = &states[MPE_GROUP_FREE_ID];
  state->kind_mask = MPE_KIND_GROUP;
  state->name = "GROUP_FREE";

  state = &states[MPE_GROUP_INCL_ID];
  state->kind_mask = MPE_KIND_GROUP;
  state->name = "GROUP_INCL";

  state = &states[MPE_GROUP_INTERSECTION_ID];
  state->kind_mask = MPE_KIND_GROUP;
  state->name = "GROUP_INTERSECTION";

  state = &states[MPE_GROUP_RANK_ID];
  state->kind_mask = MPE_KIND_GROUP;
  state->name = "GROUP_RANK";

  state = &states[MPE_GROUP_RANGE_EXCL_ID];
  state->kind_mask = MPE_KIND_GROUP;
  state->name = "GROUP_RANGE_EXCL";

  state = &states[MPE_GROUP_RANGE_INCL_ID];
  state->kind_mask = MPE_KIND_GROUP;
  state->name = "GROUP_RANGE_INCL";

  state = &states[MPE_GROUP_SIZE_ID];
  state->kind_mask = MPE_KIND_GROUP;
  state->name = "GROUP_SIZE";

  state = &states[MPE_GROUP_TRANSLATE_RANKS_ID];
  state->kind_mask = MPE_KIND_GROUP;
  state->name = "GROUP_TRANSLATE_RANKS";

  state = &states[MPE_GROUP_UNION_ID];
  state->kind_mask = MPE_KIND_GROUP;
  state->name = "GROUP_UNION";

  state = &states[MPE_INTERCOMM_CREATE_ID];
  state->kind_mask = MPE_KIND_COMM;
  state->name = "INTERCOMM_CREATE";

  state = &states[MPE_INTERCOMM_MERGE_ID];
  state->kind_mask = MPE_KIND_COMM;
  state->name = "INTERCOMM_MERGE";

  state = &states[MPE_KEYVAL_CREATE_ID];
  state->kind_mask = MPE_KIND_ATTR;
  state->name = "KEYVAL_CREATE";

  state = &states[MPE_KEYVAL_FREE_ID];
  state->kind_mask = MPE_KIND_ATTR;
  state->name = "KEYVAL_FREE";

  state = &states[MPE_ABORT_ID];
  state->kind_mask = MPE_KIND_ENV;
  state->name = "ABORT";

  state = &states[MPE_ERROR_CLASS_ID];
  state->kind_mask = MPE_KIND_ENV;
  state->name = "ERROR_CLASS";

  state = &states[MPE_ERRHANDLER_CREATE_ID];
  state->kind_mask = MPE_KIND_ENV;
  state->name = "ERRHANDLER_CREATE";

  state = &states[MPE_ERRHANDLER_FREE_ID];
  state->kind_mask = MPE_KIND_ENV;
  state->name = "ERRHANDLER_FREE";

  state = &states[MPE_ERRHANDLER_GET_ID];
  state->kind_mask = MPE_KIND_ENV;
  state->name = "ERRHANDLER_GET";

  state = &states[MPE_ERROR_STRING_ID];
  state->kind_mask = MPE_KIND_ENV;
  state->name = "ERROR_STRING";

  state = &states[MPE_ERRHANDLER_SET_ID];
  state->kind_mask = MPE_KIND_ENV;
  state->name = "ERRHANDLER_SET";

  state = &states[MPE_GET_PROCESSOR_NAME_ID];
  state->kind_mask = MPE_KIND_ENV;
  state->name = "GET_PROCESSOR_NAME";

  state = &states[MPE_INITIALIZED_ID];
  state->kind_mask = MPE_KIND_ENV;
  state->name = "INITIALIZED";

  state = &states[MPE_WTICK_ID];
  state->kind_mask = MPE_KIND_ENV;
  state->name = "WTICK";

  state = &states[MPE_WTIME_ID];
  state->kind_mask = MPE_KIND_ENV;
  state->name = "WTIME";

  state = &states[MPE_ADDRESS_ID];
  state->kind_mask = MPE_KIND_DATATYPE;
  state->name = "ADDRESS";

  state = &states[MPE_BSEND_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "BSEND";
  state->color = "blue:gray3";

  state = &states[MPE_BSEND_INIT_ID];
  state->kind_mask = MPE_KIND_MSG_INIT;
  state->name = "BSEND_INIT";

  state = &states[MPE_BUFFER_ATTACH_ID];
  state->kind_mask = MPE_KIND_MSG_INIT;
  state->name = "BUFFER_ATTACH";

  state = &states[MPE_BUFFER_DETACH_ID];
  state->kind_mask = MPE_KIND_MSG_INIT;
  state->name = "BUFFER_DETACH";

  state = &states[MPE_CANCEL_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "CANCEL";

  state = &states[MPE_REQUEST_FREE_ID];
  state->kind_mask = MPE_KIND_MSG_INIT;
  state->name = "REQUEST_FREE";

  state = &states[MPE_RECV_INIT_ID];
  state->kind_mask = MPE_KIND_MSG_INIT;
  state->name = "RECV_INIT";

  state = &states[MPE_SEND_INIT_ID];
  state->kind_mask = MPE_KIND_MSG_INIT;
  state->name = "SEND_INIT";

  state = &states[MPE_GET_ELEMENTS_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "GET_ELEMENTS";

  state = &states[MPE_GET_COUNT_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "GET_COUNT";

  state = &states[MPE_IBSEND_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "IBSEND";

  state = &states[MPE_IPROBE_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "IPROBE";
  state->color = "seagreen:gray";

  state = &states[MPE_IRECV_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "IRECV";
  state->color = "springgreen:gray";

  state = &states[MPE_IRSEND_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "IRSEND";

  state = &states[MPE_ISEND_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "ISEND";
  state->color = "skyblue:gray";
  
  state = &states[MPE_ISSEND_ID];
  state->kind_mask = MPE_ISSEND_ID;
  state->name = "ISSEND";
  state->color = "seagreen:gray";
  
  state = &states[MPE_PACK_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "PACK";
  
  state = &states[MPE_PACK_SIZE_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "PACK_SIZE";

  state = &states[MPE_PROBE_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "PROBE";
  state->color = "seagreen:gray";
  
  state = &states[MPE_RECV_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "RECV";
  state->color = "green:light_gray";
  
  state = &states[MPE_RSEND_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "RSEND";

  state = &states[MPE_RSEND_INIT_ID];
  state->kind_mask = MPE_KIND_MSG_INIT;
  state->name = "RSEND_INIT";

  state = &states[MPE_SEND_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "SEND";
  state->color = "blue:gray3";

  state = &states[MPE_SENDRECV_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "SENDRECV";
  state->color = "seagreen:gray";
  
  state = &states[MPE_SENDRECV_REPLACE_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "SENDRECV_REPLACE";
  state->color = "seagreen:gray";

  state = &states[MPE_SSEND_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "SSEND";
  state->color = "deepskyblue:gray";

  state = &states[MPE_SSEND_INIT_ID];
  state->kind_mask = MPE_KIND_MSG_INIT;
  state->name = "SSEND_INIT";

  state = &states[MPE_START_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "START";

  state = &states[MPE_STARTALL_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "STARTALL";
  
  state = &states[MPE_TEST_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "TEST";
  state->color = "orange:gray";

  state = &states[MPE_TESTALL_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "TESTALL";
  state->color = "orange:gray";

  state = &states[MPE_TESTANY_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "TESTANY";
  state->color = "orange:gray";

  state = &states[MPE_TEST_CANCELLED_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "TEST_CANCELLED";

  state = &states[MPE_TESTSOME_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "TESTSOME";
  state->color = "orange:gray";

  state = &states[MPE_TYPE_COMMIT_ID];
  state->kind_mask = MPE_KIND_DATATYPE;
  state->name = "TYPE_COMMIT";

  state = &states[MPE_TYPE_CONTIGUOUS_ID];
  state->kind_mask = MPE_KIND_DATATYPE;
  state->name = "TYPE_CONTIGUOUS";

  state = &states[MPE_TYPE_EXTENT_ID];
  state->kind_mask = MPE_KIND_DATATYPE;
  state->name = "TYPE_EXTENT";

  state = &states[MPE_TYPE_FREE_ID];
  state->kind_mask = MPE_KIND_DATATYPE;
  state->name = "TYPE_FREE";

  state = &states[MPE_TYPE_HINDEXED_ID];
  state->kind_mask = MPE_KIND_DATATYPE;
  state->name = "TYPE_HINDEXED";

  state = &states[MPE_TYPE_INDEXED_ID];
  state->kind_mask = MPE_KIND_DATATYPE;
  state->name = "TYPE_INDEXED";

  state = &states[MPE_TYPE_HVECTOR_ID];
  state->kind_mask = MPE_KIND_DATATYPE;
  state->name = "TYPE_HVECTOR";

  state = &states[MPE_TYPE_LB_ID];
  state->kind_mask = MPE_KIND_DATATYPE;
  state->name = "TYPE_LB";

  state = &states[MPE_TYPE_SIZE_ID];
  state->kind_mask = MPE_KIND_DATATYPE;
  state->name = "TYPE_SIZE";

  state = &states[MPE_TYPE_STRUCT_ID];
  state->kind_mask = MPE_KIND_DATATYPE;
  state->name = "TYPE_STRUCT";

  state = &states[MPE_TYPE_UB_ID];
  state->kind_mask = MPE_KIND_DATATYPE;
  state->name = "TYPE_UB";

  state = &states[MPE_TYPE_VECTOR_ID];
  state->kind_mask = MPE_KIND_DATATYPE;
  state->name = "TYPE_VECTOR";

  state = &states[MPE_UNPACK_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "UNPACK";

  state = &states[MPE_WAIT_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "WAIT";
  state->color = "red:black";

  state = &states[MPE_WAITALL_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "WAITALL";
  state->color = "OrangeRed:gray";

  state = &states[MPE_WAITANY_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "WAITANY";
  state->color = "coral:gray";

  state = &states[MPE_WAITSOME_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "WAITSOME";
  state->color = "red:black";
  
  state = &states[MPE_CART_COORDS_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "CART_COORDS";

  state = &states[MPE_CART_CREATE_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "CART_CREATE";

  state = &states[MPE_CART_GET_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "CART_GET";

  state = &states[MPE_CART_MAP_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "CART_MAP";

  state = &states[MPE_CART_RANK_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "CART_RANK";

  state = &states[MPE_CART_SHIFT_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "CART_SHIFT";

  state = &states[MPE_CART_SUB_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "CART_SUB";

  state = &states[MPE_CARTDIM_GET_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "CARTDIM_GET";

  state = &states[MPE_DIMS_CREATE_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "DIMS_CREATE";

  state = &states[MPE_GRAPH_CREATE_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "GRAPH_CREATE";

  state = &states[MPE_GRAPH_GET_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "GRAPH_GET";

  state = &states[MPE_GRAPH_MAP_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "GRAPH_MAP";

  state = &states[MPE_GRAPH_NEIGHBORS_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "GRAPH_NEIGHBORS";

  state = &states[MPE_GRAPH_NEIGHBORS_COUNT_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "GRAPH_NEIGHBORS_COUNT";

  state = &states[MPE_GRAPHDIMS_GET_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "GRAPHDIMS_GET";

  state = &states[MPE_TOPO_TEST_ID];
  state->kind_mask = MPE_KIND_TOPO;
  state->name = "TOPO_TEST";

  state = &states[MPE_RECV_IDLE_ID];
  state->kind_mask = MPE_KIND_MSG;
  state->name = "RECV_IDLE";

  /* Set default logfilename */  
/*  sprintf( logFileName_0, "%s_profile.log", (*argv)[0] ); */
  sprintf( logFileName_0, "%s", (*argv)[0] );

  /* Enable the basic states */
  for (i=0; i<MPE_MAX_STATES; i++) {
  	if ((states[i].kind_mask & allow_mask) != 0) states[i].is_active = 1;
  }

  rq_init( requests_avail_0 );

  MPE_Start_log();
  trace_on = 1;

  return returnVal;
}

int  MPI_Initialized( flag )
int * flag;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Initialized - prototyping replacement for MPI_Initialized
    Log the beginning and ending of the time spent in MPI_Initialized calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_INITIALIZED_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Initialized( flag );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

#ifdef FOO
/* Use the regular routines for these.  Note that the state logging needs
   MPI_Wtime; make sure that it uses PMPI_Wtime if you use these
   */
double  MPI_Wtick(  )
{
  double  returnVal;
  register MPE_State *state;

/*
    MPI_Wtick - prototyping replacement for MPI_Wtick
    Log the beginning and ending of the time spent in MPI_Wtick calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_WTICK_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Wtick(  );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

double  MPI_Wtime(  )
{
  double  returnVal;
  register MPE_State *state;

/*
    MPI_Wtime - prototyping replacement for MPI_Wtime
    Log the beginning and ending of the time spent in MPI_Wtime calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_WTIME_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Wtime(  );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}
#endif

int  MPI_Address( location, address )
void * location;
MPI_Aint * address;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Address - prototyping replacement for MPI_Address
    Log the beginning and ending of the time spent in MPI_Address calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ADDRESS_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Address( location, address );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Bsend( buf, count, datatype, dest, tag, comm )
void * buf;
int count;
MPI_Datatype datatype;
int dest;
int tag;
MPI_Comm comm;
{
  int  returnVal;
  int  size;
  register MPE_State *state;

/*
    MPI_Bsend - prototyping replacement for MPI_Bsend
    Log the beginning and ending of the time spent in MPI_Bsend calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_BSEND_ID,comm);
  PMPI_Type_size( datatype, &size );
  MPE_LOG_DO(MPE_Log_send( dest, tag, count * size ));
  
  returnVal = PMPI_Bsend( buf, count, datatype, dest, tag, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Bsend_init( buf, count, datatype, dest, tag, comm, request )
void * buf;
int count;
MPI_Datatype datatype;
int dest;
int tag;
MPI_Comm comm;
MPI_Request * request;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Bsend_init - prototyping replacement for MPI_Bsend_init
    Log the beginning and ending of the time spent in MPI_Bsend_init calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_BSEND_INIT_ID,comm);
  
  returnVal = PMPI_Bsend_init( buf, count, datatype, dest, tag, comm, request );

  if (dest != MPI_PROC_NULL) {
      MPE_Add_send_req( count, datatype, dest, tag, *request, 1 );
      /* Note not started yet ... */
      }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Buffer_attach( buffer, size )
void * buffer;
int size;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Buffer_attach - prototyping replacement for MPI_Buffer_attach
    Log the beginning and ending of the time spent in MPI_Buffer_attach calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_BUFFER_ATTACH_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Buffer_attach( buffer, size );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Buffer_detach( buffer, size )
void * buffer;
int * size;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Buffer_detach - prototyping replacement for MPI_Buffer_detach
    Log the beginning and ending of the time spent in MPI_Buffer_detach calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_BUFFER_DETACH_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Buffer_detach( buffer, size );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Cancel( request )
MPI_Request * request;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Cancel - prototyping replacement for MPI_Cancel
    Log the beginning and ending of the time spent in MPI_Cancel calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_CANCEL_ID,MPI_COMM_NULL);
  
  MPE_Cancel_req( *request );

  returnVal = PMPI_Cancel( request );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Request_free( request )
MPI_Request * request;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Request_free - prototyping replacement for MPI_Request_free
    Log the beginning and ending of the time spent in MPI_Request_free calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_REQUEST_FREE_ID,MPI_COMM_NULL);
  
  MPE_Remove_req( *request );

  returnVal = PMPI_Request_free( request );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Recv_init( buf, count, datatype, source, tag, comm, request )
void * buf;
int count;
MPI_Datatype datatype;
int source;
int tag;
MPI_Comm comm;
MPI_Request * request;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Recv_init - prototyping replacement for MPI_Recv_init
    Log the beginning and ending of the time spent in MPI_Recv_init calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_RECV_INIT_ID,comm);
  
  returnVal = PMPI_Recv_init( buf, count, datatype, source, tag, comm, request );

  if (returnVal == MPI_SUCCESS && source != MPI_PROC_NULL) {
      MPE_Add_recv_req( count, datatype, source, tag, *request, 1 );
      /* Not started yet ... */
      }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Send_init( buf, count, datatype, dest, tag, comm, request )
void * buf;
int count;
MPI_Datatype datatype;
int dest;
int tag;
MPI_Comm comm;
MPI_Request * request;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Send_init - prototyping replacement for MPI_Send_init
    Log the beginning and ending of the time spent in MPI_Send_init calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_SEND_INIT_ID,comm);
  
  returnVal = PMPI_Send_init( buf, count, datatype, dest, tag, comm, request );

  if (dest != MPI_PROC_NULL) {
      MPE_Add_send_req( count, datatype, dest, tag, *request, 1 );
      /* Note not started yet ... */
      }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Get_elements( status, datatype, elements )
MPI_Status * status;
MPI_Datatype datatype;
int * elements;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Get_elements - prototyping replacement for MPI_Get_elements
    Log the beginning and ending of the time spent in MPI_Get_elements calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GET_ELEMENTS_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Get_elements( status, datatype, elements );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Get_count( status, datatype, count )
MPI_Status * status;
MPI_Datatype datatype;
int * count;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Get_count - prototyping replacement for MPI_Get_count
    Log the beginning and ending of the time spent in MPI_Get_count calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GET_COUNT_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Get_count( status, datatype, count );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Ibsend( buf, count, datatype, dest, tag, comm, request )
void * buf;
int count;
MPI_Datatype datatype;
int dest;
int tag;
MPI_Comm comm;
MPI_Request * request;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Ibsend - prototyping replacement for MPI_Ibsend
    Log the beginning and ending of the time spent in MPI_Ibsend calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_IBSEND_ID,comm);
  
  returnVal = PMPI_Ibsend( buf, count, datatype, dest, tag, comm, request );

  if (dest != MPI_PROC_NULL) {
      MPE_Add_send_req( count, datatype, dest, tag, *request, 0 );
      }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Iprobe( source, tag, comm, flag, status )
int source;
int tag;
MPI_Comm comm;
int * flag;
MPI_Status * status;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Iprobe - prototyping replacement for MPI_Iprobe
    Log the beginning and ending of the time spent in MPI_Iprobe calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_IPROBE_ID,comm);
  
  returnVal = PMPI_Iprobe( source, tag, comm, flag, status );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Irecv( buf, count, datatype, source, tag, comm, request )
void * buf;
int count;
MPI_Datatype datatype;
int source;
int tag;
MPI_Comm comm;
MPI_Request * request;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Irecv - prototyping replacement for MPI_Irecv
    Log the beginning and ending of the time spent in MPI_Irecv calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_IRECV_ID,comm);
  
  returnVal = PMPI_Irecv( buf, count, datatype, source, tag, comm, request );

  if (returnVal == MPI_SUCCESS && source != MPI_PROC_NULL) {
      MPE_Add_recv_req( count, datatype, source, tag, *request, 0 );
      }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Irsend( buf, count, datatype, dest, tag, comm, request )
void * buf;
int count;
MPI_Datatype datatype;
int dest;
int tag;
MPI_Comm comm;
MPI_Request * request;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Irsend - prototyping replacement for MPI_Irsend
    Log the beginning and ending of the time spent in MPI_Irsend calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_IRSEND_ID,comm);

  returnVal = PMPI_Irsend( buf, count, datatype, dest, tag, comm, request );

  if (dest != MPI_PROC_NULL) {
      MPE_Add_send_req( count, datatype, dest, tag, *request, 0 );
      }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Isend( buf, count, datatype, dest, tag, comm, request )
void * buf;
int count;
MPI_Datatype datatype;
int dest;
int tag;
MPI_Comm comm;
MPI_Request * request;
{
  int  returnVal;
  int  size;
  register MPE_State *state;
/*
    MPI_Isend - prototyping replacement for MPI_Isend
    Log the beginning and ending of the time spent in MPI_Isend calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ISEND_ID,comm);
  PMPI_Type_size( datatype, &size );  	         
  MPE_LOG_DO(MPE_Log_send( dest, tag, size * count ));
  returnVal = PMPI_Isend( buf, count, datatype, dest, tag, comm, request );

  if (dest != MPI_PROC_NULL) {
      MPE_Add_send_req( count, datatype, dest, tag, *request, 0 );
  }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Issend( buf, count, datatype, dest, tag, comm, request )
void * buf;
int count;
MPI_Datatype datatype;
int dest;
int tag;
MPI_Comm comm;
MPI_Request * request;
{
  int  returnVal;
  int  size;
  register MPE_State *state;

/*
    MPI_Issend - prototyping replacement for MPI_Issend
    Log the beginning and ending of the time spent in MPI_Issend calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_ISSEND_ID,comm);
  PMPI_Type_size( datatype, &size );  	         
  MPE_LOG_DO(MPE_Log_send( dest, tag, size * count ));
  
  returnVal = PMPI_Issend( buf, count, datatype, dest, tag, comm, request );

  if (dest != MPI_PROC_NULL) {
      MPE_Add_send_req( count, datatype, dest, tag, *request, 0 );
      }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Pack( inbuf, incount, type, outbuf, outcount, position, comm )
void * inbuf;
int incount;
MPI_Datatype type;
void * outbuf;
int outcount;
int * position;
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Pack - prototyping replacement for MPI_Pack
    Log the beginning and ending of the time spent in MPI_Pack calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_PACK_ID,comm);
  
  returnVal = PMPI_Pack( inbuf, incount, type, outbuf, outcount, position, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Pack_size( incount, datatype, comm, size )
int incount;
MPI_Datatype datatype;
MPI_Comm comm;
int * size;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Pack_size - prototyping replacement for MPI_Pack_size
    Log the beginning and ending of the time spent in MPI_Pack_size calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_PACK_SIZE_ID,comm);
  
  returnVal = PMPI_Pack_size( incount, datatype, comm, size );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Probe( source, tag, comm, status )
int source;
int tag;
MPI_Comm comm;
MPI_Status * status;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Probe - prototyping replacement for MPI_Probe
    Log the beginning and ending of the time spent in MPI_Probe calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_PROBE_ID,comm);
  
  returnVal = PMPI_Probe( source, tag, comm, status );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Recv( buf, count, datatype, source, tag, comm, status )
void * buf;
int count;
MPI_Datatype datatype;
int source;
int tag;
MPI_Comm comm;
MPI_Status * status;
{
  int  returnVal, acount;
  register MPE_State *state;

/*
    MPI_Recv - prototyping replacement for MPI_Recv
    Log the beginning and ending of the time spent in MPI_Recv calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_RECV_ID,comm);
  
  returnVal = PMPI_Recv( buf, count, datatype, source, tag, comm, status );

  if (returnVal == MPI_SUCCESS) {
      PMPI_Get_count( status, MPI_BYTE, &acount );
      MPE_LOG_DO(MPE_Log_receive(  status->MPI_SOURCE, status->MPI_TAG, acount ));
      }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Rsend( buf, count, datatype, dest, tag, comm )
void * buf;
int count;
MPI_Datatype datatype;
int dest;
int tag;
MPI_Comm comm;
{
  int  returnVal;
  int  size;
  register MPE_State *state;

/*
    MPI_Rsend - prototyping replacement for MPI_Rsend
    Log the beginning and ending of the time spent in MPI_Rsend calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_RSEND_ID,comm);

  PMPI_Type_size( datatype, &size );
  MPE_LOG_DO(MPE_Log_send( dest, tag, count * size ));

  returnVal = PMPI_Rsend( buf, count, datatype, dest, tag, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Rsend_init( buf, count, datatype, dest, tag, comm, request )
void * buf;
int count;
MPI_Datatype datatype;
int dest;
int tag;
MPI_Comm comm;
MPI_Request * request;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Rsend_init - prototyping replacement for MPI_Rsend_init
    Log the beginning and ending of the time spent in MPI_Rsend_init calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_RSEND_INIT_ID,comm);
  
  returnVal = PMPI_Rsend_init( buf, count, datatype, dest, tag, comm, request );

  if (dest != MPI_PROC_NULL) {
      MPE_Add_send_req( count, datatype, dest, tag, *request, 1 );
      /* Note not started yet ... */
      }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Send( buf, count, datatype, dest, tag, comm )
void * buf;
int count;
MPI_Datatype datatype;
int dest;
int tag;
MPI_Comm comm;
{
  int  returnVal;
  int  size;
  register MPE_State *state;
/*
    MPI_Send - prototyping replacement for MPI_Send
    Log the beginning and ending of the time spent in MPI_Send calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_SEND_ID,comm);
  PMPI_Type_size( datatype, &size );
  MPE_LOG_DO(MPE_Log_send( dest, tag, size * count ));

  returnVal = PMPI_Send( buf, count, datatype, dest, tag, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Sendrecv( sendbuf, sendcount, sendtype, dest, sendtag, 
		  recvbuf, recvcount, recvtype, source, recvtag, comm, status )
void * sendbuf;
int sendcount;
MPI_Datatype sendtype;
int dest;
int sendtag;
void * recvbuf;
int recvcount;
MPI_Datatype recvtype;
int source;
int recvtag;
MPI_Comm comm;
MPI_Status * status;
{
  int  returnVal;
  int  acount, sendsize;
  register MPE_State *state;

/*
    MPI_Sendrecv - prototyping replacement for MPI_Sendrecv
    Log the beginning and ending of the time spent in MPI_Sendrecv calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_SENDRECV_ID,comm);

  returnVal = PMPI_Sendrecv( sendbuf, sendcount, sendtype, dest, sendtag, 
			     recvbuf, recvcount, recvtype, source, recvtag, 
			     comm, status );

  if (returnVal == MPI_SUCCESS) {
      PMPI_Type_size( sendtype, &sendsize );
      MPE_LOG_DO(MPE_Log_send( dest, sendtag, sendcount * sendsize ));
      PMPI_Get_count( status, MPI_BYTE, &acount );
      MPE_LOG_DO(MPE_Log_receive( status->MPI_SOURCE, status->MPI_TAG, acount ));
      }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Sendrecv_replace( buf, count, datatype, dest, sendtag, source, 
			   recvtag, comm, status )
void * buf;
int count;
MPI_Datatype datatype;
int dest;
int sendtag;
int source;
int recvtag;
MPI_Comm comm;
MPI_Status * status;
{
  int  returnVal;
  int  acount, sendsize;
  register MPE_State *state;
/*
    MPI_Sendrecv_replace - prototyping replacement for MPI_Sendrecv_replace
    Log the beginning and ending of the time spent in MPI_Sendrecv_replace calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_SENDRECV_REPLACE_ID,comm);
  
  returnVal = PMPI_Sendrecv_replace( buf, count, datatype, dest, 
				     sendtag, source, recvtag, comm, status );

  if (returnVal == MPI_SUCCESS) {
      PMPI_Type_size( datatype, &sendsize );
      MPE_LOG_DO(MPE_Log_send( dest, sendtag, count * sendsize ));
      PMPI_Get_count( status, MPI_BYTE, &acount );
      MPE_LOG_DO(MPE_Log_receive( status->MPI_SOURCE, status->MPI_TAG, acount ));
      }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Ssend( buf, count, datatype, dest, tag, comm )
void * buf;
int count;
MPI_Datatype datatype;
int dest;
int tag;
MPI_Comm comm;
{
  int  returnVal, size;
  register MPE_State *state;

/*
    MPI_Ssend - prototyping replacement for MPI_Ssend
    Log the beginning and ending of the time spent in MPI_Ssend calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_SSEND_ID,comm);
  
  returnVal = PMPI_Ssend( buf, count, datatype, dest, tag, comm );

  if (!returnVal){
      PMPI_Type_size( datatype, &size );
      MPE_LOG_DO(MPE_Log_send(  dest, tag, count * size ));
  }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Ssend_init( buf, count, datatype, dest, tag, comm, request )
void * buf;
int count;
MPI_Datatype datatype;
int dest;
int tag;
MPI_Comm comm;
MPI_Request * request;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Ssend_init - prototyping replacement for MPI_Ssend_init
    Log the beginning and ending of the time spent in MPI_Ssend_init calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_SSEND_INIT_ID,comm);
  
  returnVal = PMPI_Ssend_init( buf, count, datatype, dest, tag, comm, request );

  if (dest != MPI_PROC_NULL) {
      MPE_Add_send_req( count, datatype, dest, tag, *request, 1 );
      /* Note not started yet ... */
      }

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Start( request )
MPI_Request * request;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Start - prototyping replacement for MPI_Start
    Log the beginning and ending of the time spent in MPI_Start calls.
*/

  MPE_LOG_STATE_BEGIN( MPE_START_ID, MPI_COMM_NULL );
  
  returnVal = PMPI_Start( request );

  MPE_LOG_STATE_END( MPI_COMM_NULL );
  MPE_Start_req( *request, state );

  return returnVal;
}

int  MPI_Startall( count, array_of_requests )
int count;
MPI_Request * array_of_requests;
{
  int  returnVal;
  int  i;
  register MPE_State *state;
/*
    MPI_Startall - prototyping replacement for MPI_Startall
    Log the beginning and ending of the time spent in MPI_Startall calls.
*/

  MPE_LOG_STATE_BEGIN( MPE_STARTALL_ID, MPI_COMM_NULL );
  
  returnVal = PMPI_Startall( count, array_of_requests );

  for (i=0; i<count; i++)
      MPE_Start_req( array_of_requests[i], state );

  MPE_LOG_STATE_END( MPI_COMM_NULL );

  return returnVal;
}

int   MPI_Test( request, flag, status )
MPI_Request * request;
int * flag;
MPI_Status * status;
{
  int   returnVal;
  MPI_Request lreq = *request;
  register MPE_State *state;
/*
    MPI_Test - prototyping replacement for MPI_Test
    Log the beginning and ending of the time spent in MPI_Test calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TEST_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Test( request, flag, status );

  if (*flag) 
      MPE_ProcessWaitTest( lreq, status, "MPI_Test", state );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Testall( count, array_of_requests, flag, array_of_statuses )
int count;
MPI_Request * array_of_requests;
int * flag;
MPI_Status * array_of_statuses;
{
  int  returnVal;
  int  i;
  register MPE_State *state;

/*
    MPI_Testall - prototyping replacement for MPI_Testall
    Log the beginning and ending of the time spent in MPI_Testall calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TESTALL_ID,MPI_COMM_NULL);
  
  if (count < MPE_MAX_REQUESTS) {
      for (i=0; i<count; i++) 
	  req[i] = array_of_requests[i];
  }

  returnVal = PMPI_Testall( count, array_of_requests, flag, array_of_statuses );

  if (*flag && count < MPE_MAX_REQUESTS) {
    for (i=0; i < count; i++) {
      MPE_ProcessWaitTest( req[i], &array_of_statuses[i], "MPI_Testall",
			   state );
    }
  }

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Testany( count, array_of_requests, index, flag, status )
int count;
MPI_Request * array_of_requests;
int * index;
int * flag;
MPI_Status * status;
{
  int  returnVal;
  register MPE_State *state;
  int i;

/*
    MPI_Testany - prototyping replacement for MPI_Testany
    Log the beginning and ending of the time spent in MPI_Testany calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TESTANY_ID,MPI_COMM_NULL);
  
  if (count < MPE_MAX_REQUESTS) {
      for (i=0; i<count; i++) 
	  req[i] = array_of_requests[i];
  }

  returnVal = PMPI_Testany( count, array_of_requests, index, flag, status );

  if (*flag && count < MPE_MAX_REQUESTS) 
      MPE_ProcessWaitTest( req[*index], status, "MPI_Testany", state );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Test_cancelled( status, flag )
MPI_Status * status;
int * flag;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Test_cancelled - prototyping replacement for MPI_Test_cancelled
    Log the beginning and ending of the time spent in MPI_Test_cancelled calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TEST_CANCELLED_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Test_cancelled( status, flag );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Testsome( incount, array_of_requests, outcount, 
		   array_of_indices, array_of_statuses )
int incount;
MPI_Request * array_of_requests;
int * outcount;
int * array_of_indices;
MPI_Status * array_of_statuses;
{
  int  returnVal;
  int  i;
  register MPE_State *state;

/*
    MPI_Testsome - prototyping replacement for MPI_Testsome
    Log the beginning and ending of the time spent in MPI_Testsome calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TESTSOME_ID,MPI_COMM_NULL);
  
  if (incount < MPE_MAX_REQUESTS) {
      for (i=0; i<incount; i++) 
	  req[i] = array_of_requests[i];
  }

  returnVal = PMPI_Testsome( incount, array_of_requests, outcount, 
			     array_of_indices, array_of_statuses );

  if (incount < MPE_MAX_REQUESTS) {
      for (i=0; i < *outcount; i++) {
	  MPE_ProcessWaitTest( req[array_of_indices[i]], 
	       &array_of_statuses[array_of_indices[i]], "MPI_Testsome",
			       state );
      }
  }

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Type_commit( datatype )
MPI_Datatype * datatype;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Type_commit - prototyping replacement for MPI_Type_commit
    Log the beginning and ending of the time spent in MPI_Type_commit calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TYPE_COMMIT_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Type_commit( datatype );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Type_contiguous( count, old_type, newtype )
int count;
MPI_Datatype old_type;
MPI_Datatype * newtype;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Type_contiguous - prototyping replacement for MPI_Type_contiguous
    Log the beginning and ending of the time spent in MPI_Type_contiguous calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TYPE_CONTIGUOUS_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Type_contiguous( count, old_type, newtype );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Type_extent( datatype, extent )
MPI_Datatype datatype;
MPI_Aint * extent;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Type_extent - prototyping replacement for MPI_Type_extent
    Log the beginning and ending of the time spent in MPI_Type_extent calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TYPE_EXTENT_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Type_extent( datatype, extent );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Type_free( datatype )
MPI_Datatype * datatype;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Type_free - prototyping replacement for MPI_Type_free
    Log the beginning and ending of the time spent in MPI_Type_free calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TYPE_FREE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Type_free( datatype );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Type_hindexed( count, blocklens, indices, old_type, newtype )
int count;
int * blocklens;
MPI_Aint * indices;
MPI_Datatype old_type;
MPI_Datatype * newtype;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Type_hindexed - prototyping replacement for MPI_Type_hindexed
    Log the beginning and ending of the time spent in MPI_Type_hindexed calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TYPE_HINDEXED_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Type_hindexed( count, blocklens, indices, old_type, newtype );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Type_hvector( count, blocklen, stride, old_type, newtype )
int count;
int blocklen;
MPI_Aint stride;
MPI_Datatype old_type;
MPI_Datatype * newtype;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Type_hvector - prototyping replacement for MPI_Type_hvector
    Log the beginning and ending of the time spent in MPI_Type_hvector calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TYPE_HVECTOR_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Type_hvector( count, blocklen, stride, old_type, newtype );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Type_indexed( count, blocklens, indices, old_type, newtype )
int count;
int * blocklens;
int * indices;
MPI_Datatype old_type;
MPI_Datatype * newtype;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Type_indexed - prototyping replacement for MPI_Type_indexed
    Log the beginning and ending of the time spent in MPI_Type_indexed calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TYPE_INDEXED_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Type_indexed( count, blocklens, indices, old_type, newtype );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Type_lb( datatype, displacement )
MPI_Datatype datatype;
MPI_Aint * displacement;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Type_lb - prototyping replacement for MPI_Type_lb
    Log the beginning and ending of the time spent in MPI_Type_lb calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TYPE_LB_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Type_lb( datatype, displacement );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Type_size( datatype, size )
MPI_Datatype datatype;
int          * size;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Type_size - prototyping replacement for MPI_Type_size
    Log the beginning and ending of the time spent in MPI_Type_size calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TYPE_SIZE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Type_size( datatype, size );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Type_struct( count, blocklens, indices, old_types, newtype )
int count;
int * blocklens;
MPI_Aint * indices;
MPI_Datatype * old_types;
MPI_Datatype * newtype;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Type_struct - prototyping replacement for MPI_Type_struct
    Log the beginning and ending of the time spent in MPI_Type_struct calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TYPE_STRUCT_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Type_struct( count, blocklens, indices, old_types, newtype );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Type_ub( datatype, displacement )
MPI_Datatype datatype;
MPI_Aint * displacement;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Type_ub - prototyping replacement for MPI_Type_ub
    Log the beginning and ending of the time spent in MPI_Type_ub calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TYPE_UB_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Type_ub( datatype, displacement );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Type_vector( count, blocklen, stride, old_type, newtype )
int count;
int blocklen;
int stride;
MPI_Datatype old_type;
MPI_Datatype * newtype;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Type_vector - prototyping replacement for MPI_Type_vector
    Log the beginning and ending of the time spent in MPI_Type_vector calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TYPE_VECTOR_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Type_vector( count, blocklen, stride, old_type, newtype );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Unpack( inbuf, insize, position, outbuf, outcount, type, comm )
void * inbuf;
int insize;
int * position;
void * outbuf;
int outcount;
MPI_Datatype type;
MPI_Comm comm;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Unpack - prototyping replacement for MPI_Unpack
    Log the beginning and ending of the time spent in MPI_Unpack calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_UNPACK_ID,comm);
  
  returnVal = PMPI_Unpack( inbuf, insize, position, outbuf, outcount, type, comm );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Wait( request, status )
MPI_Request * request;
MPI_Status * status;
{
  int   returnVal;
  register MPE_State *state;
  MPI_Request lreq = *request;

/*
    MPI_Wait - prototyping replacement for MPI_Wait
    Log the beginning and ending of the time spent in MPI_Wait calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_WAIT_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Wait( request, status );

  MPE_ProcessWaitTest( lreq, status, "MPI_Wait", state );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Waitall( count, array_of_requests, array_of_statuses )
int count;
MPI_Request * array_of_requests;
MPI_Status * array_of_statuses;
{
  int  returnVal;
  int  i;
  register MPE_State *state;

/*
    MPI_Waitall - prototyping replacement for MPI_Waitall
    Log the beginning and ending of the time spent in MPI_Waitall calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_WAITALL_ID,MPI_COMM_NULL);
  
  if (count < MPE_MAX_REQUESTS) {
      for (i=0; i<count; i++) 
	  req[i] = array_of_requests[i];
  }

  returnVal = PMPI_Waitall( count, array_of_requests, array_of_statuses );

  if (count < MPE_MAX_REQUESTS) {
      for (i=0; i < count; i++) {
	  MPE_ProcessWaitTest( req[i], &array_of_statuses[i], "MPI_Waitall",
			       state );
      }
  }

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Waitany( count, array_of_requests, index, status )
int count;
MPI_Request * array_of_requests;
int * index;
MPI_Status * status;
{
  int  returnVal;
  register MPE_State *state;
  int i;

/*
    MPI_Waitany - prototyping replacement for MPI_Waitany
    Log the beginning and ending of the time spent in MPI_Waitany calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_WAITANY_ID,MPI_COMM_NULL);
  
  if (count < MPE_MAX_REQUESTS) {
      for (i=0; i<count; i++) 
	  req[i] = array_of_requests[i];
  }

  returnVal = PMPI_Waitany( count, array_of_requests, index, status );

  if (count < MPE_MAX_REQUESTS) {
      MPE_ProcessWaitTest( req[*index], status, "MPI_Waitany", state );
  }

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int  MPI_Waitsome( incount, array_of_requests, outcount, array_of_indices, array_of_statuses )
int incount;
MPI_Request * array_of_requests;
int * outcount;
int * array_of_indices;
MPI_Status * array_of_statuses;
{
  int  returnVal;
  int  i;
  register MPE_State *state;

/*
    MPI_Waitsome - prototyping replacement for MPI_Waitsome
    Log the beginning and ending of the time spent in MPI_Waitsome calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_WAITSOME_ID,MPI_COMM_NULL);
  
  if (incount < MPE_MAX_REQUESTS) {
      for (i=0; i<incount; i++) 
	  req[i] = array_of_requests[i];
  }

  returnVal = PMPI_Waitsome( incount, array_of_requests, outcount, 
			     array_of_indices, array_of_statuses );

  if (incount < MPE_MAX_REQUESTS) {
      for (i=0; i < *outcount; i++) {
	  MPE_ProcessWaitTest( req[array_of_indices[i]],
			        &array_of_statuses[array_of_indices[i]],
			        "MPI_Waitsome", state );
      }
  }

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Cart_coords( comm, rank, maxdims, coords )
MPI_Comm comm;
int rank;
int maxdims;
int * coords;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Cart_coords - prototyping replacement for MPI_Cart_coords
    Log the beginning and ending of the time spent in MPI_Cart_coords calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_CART_COORDS_ID,comm);
  
  returnVal = PMPI_Cart_coords( comm, rank, maxdims, coords );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Cart_create( comm_old, ndims, dims, periods, reorder, comm_cart )
MPI_Comm comm_old;
int ndims;
int * dims;
int * periods;
int reorder;
MPI_Comm * comm_cart;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Cart_create - prototyping replacement for MPI_Cart_create
    Log the beginning and ending of the time spent in MPI_Cart_create calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_CART_CREATE_ID,comm);
  
  returnVal = PMPI_Cart_create( comm_old, ndims, dims, periods, reorder, comm_cart );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Cart_get( comm, maxdims, dims, periods, coords )
MPI_Comm comm;
int maxdims;
int * dims;
int * periods;
int * coords;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Cart_get - prototyping replacement for MPI_Cart_get
    Log the beginning and ending of the time spent in MPI_Cart_get calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_CART_GET_ID,comm);
  
  returnVal = PMPI_Cart_get( comm, maxdims, dims, periods, coords );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Cart_map( comm_old, ndims, dims, periods, newrank )
MPI_Comm comm_old;
int ndims;
int * dims;
int * periods;
int * newrank;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Cart_map - prototyping replacement for MPI_Cart_map
    Log the beginning and ending of the time spent in MPI_Cart_map calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_CART_MAP_ID,comm_old);
  
  returnVal = PMPI_Cart_map( comm_old, ndims, dims, periods, newrank );

  MPE_LOG_STATE_END(comm_old);

  return returnVal;
}

int   MPI_Cart_rank( comm, coords, rank )
MPI_Comm comm;
int * coords;
int * rank;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Cart_rank - prototyping replacement for MPI_Cart_rank
    Log the beginning and ending of the time spent in MPI_Cart_rank calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_CART_RANK_ID,comm);
  
  returnVal = PMPI_Cart_rank( comm, coords, rank );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Cart_shift( comm, direction, displ, source, dest )
MPI_Comm comm;
int direction;
int displ;
int * source;
int * dest;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Cart_shift - prototyping replacement for MPI_Cart_shift
    Log the beginning and ending of the time spent in MPI_Cart_shift calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_CART_SHIFT_ID,comm);
  
  returnVal = PMPI_Cart_shift( comm, direction, displ, source, dest );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Cart_sub( comm, remain_dims, comm_new )
MPI_Comm comm;
int * remain_dims;
MPI_Comm * comm_new;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Cart_sub - prototyping replacement for MPI_Cart_sub
    Log the beginning and ending of the time spent in MPI_Cart_sub calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_CART_SUB_ID,comm);
  
  returnVal = PMPI_Cart_sub( comm, remain_dims, comm_new );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Cartdim_get( comm, ndims )
MPI_Comm comm;
int * ndims;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Cartdim_get - prototyping replacement for MPI_Cartdim_get
    Log the beginning and ending of the time spent in MPI_Cartdim_get calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_CARTDIM_GET_ID,comm);
  
  returnVal = PMPI_Cartdim_get( comm, ndims );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int  MPI_Dims_create( nnodes, ndims, dims )
int nnodes;
int ndims;
int * dims;
{
  int  returnVal;
  register MPE_State *state;

/*
    MPI_Dims_create - prototyping replacement for MPI_Dims_create
    Log the beginning and ending of the time spent in MPI_Dims_create calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_DIMS_CREATE_ID,MPI_COMM_NULL);
  
  returnVal = PMPI_Dims_create( nnodes, ndims, dims );

  MPE_LOG_STATE_END(MPI_COMM_NULL);

  return returnVal;
}

int   MPI_Graph_create( comm_old, nnodes, index, edges, reorder, comm_graph )
MPI_Comm comm_old;
int nnodes;
int * index;
int * edges;
int reorder;
MPI_Comm * comm_graph;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Graph_create - prototyping replacement for MPI_Graph_create
    Log the beginning and ending of the time spent in MPI_Graph_create calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GRAPH_CREATE_ID,comm);
  
  returnVal = PMPI_Graph_create( comm_old, nnodes, index, edges, reorder, comm_graph );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Graph_get( comm, maxindex, maxedges, index, edges )
MPI_Comm comm;
int maxindex;
int maxedges;
int * index;
int * edges;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Graph_get - prototyping replacement for MPI_Graph_get
    Log the beginning and ending of the time spent in MPI_Graph_get calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GRAPH_GET_ID,comm);
  
  returnVal = PMPI_Graph_get( comm, maxindex, maxedges, index, edges );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Graph_map( comm_old, nnodes, index, edges, newrank )
MPI_Comm comm_old;
int nnodes;
int * index;
int * edges;
int * newrank;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Graph_map - prototyping replacement for MPI_Graph_map
    Log the beginning and ending of the time spent in MPI_Graph_map calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GRAPH_MAP_ID,comm_old);
  
  returnVal = PMPI_Graph_map( comm_old, nnodes, index, edges, newrank );

  MPE_LOG_STATE_END(comm_old);

  return returnVal;
}

int   MPI_Graph_neighbors( comm, rank, maxneighbors, neighbors )
MPI_Comm comm;
int rank;
int maxneighbors;
int * neighbors;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Graph_neighbors - prototyping replacement for MPI_Graph_neighbors
    Log the beginning and ending of the time spent in MPI_Graph_neighbors calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GRAPH_NEIGHBORS_ID,comm);
  
  returnVal = PMPI_Graph_neighbors( comm, rank, maxneighbors, neighbors );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Graph_neighbors_count( comm, rank, nneighbors )
MPI_Comm comm;
int rank;
int * nneighbors;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Graph_neighbors_count - prototyping replacement for MPI_Graph_neighbors_count
    Log the beginning and ending of the time spent in MPI_Graph_neighbors_count calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GRAPH_NEIGHBORS_COUNT_ID,comm);
  
  returnVal = PMPI_Graph_neighbors_count( comm, rank, nneighbors );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Graphdims_get( comm, nnodes, nedges )
MPI_Comm comm;
int * nnodes;
int * nedges;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Graphdims_get - prototyping replacement for MPI_Graphdims_get
    Log the beginning and ending of the time spent in MPI_Graphdims_get calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_GRAPHDIMS_GET_ID,comm);
  
  returnVal = PMPI_Graphdims_get( comm, nnodes, nedges );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

int   MPI_Topo_test( comm, top_type )
MPI_Comm comm;
int * top_type;
{
  int   returnVal;
  register MPE_State *state;

/*
    MPI_Topo_test - prototyping replacement for MPI_Topo_test
    Log the beginning and ending of the time spent in MPI_Topo_test calls.
*/

  MPE_LOG_STATE_BEGIN(MPE_TOPO_TEST_ID,comm);
  
  returnVal = PMPI_Topo_test( comm, top_type );

  MPE_LOG_STATE_END(comm);

  return returnVal;
}

/*
  level = 1 turns on tracing, 
  level = 0 turns it off.

  Still to do: in some cases, must log communicator operations even if
  logging is off.
 */
#if defined(__STDC__) || defined(__cplusplus) || defined(HAVE_PROTOTYPES) || \
    defined(PCONTROL_NEEDS_CONST)
#ifdef HAVE_NO_C_CONST
int MPI_Pcontrol( int level, ... )
#else
int MPI_Pcontrol( const int level, ... )
#endif
#else
int MPI_Pcontrol( level )
int level;
#endif
{
    trace_on = level;
    return MPI_SUCCESS;
}
