!
!    "$Id: ex1f.F,v 1.17 1998/04/20 18:11:27 balay Exp $"
!
!   Program usage:  mpirun ex1f [-help] [all PETSc options] 
!
!/*T
!   Concepts: Vectors^Using basic vector routines;
!   Routines: VecCreate(); VecDuplicate(); VecSet(); VecValid(); 
!   Routines: VecDot(); VecMDot(); VecScale(); VecNorm(); VecCopy(); VecAXPY(); 
!   Routines: VecAYPX(); VecWAXPY(); VecPointwiseMult(); VecPointwiseDivide(); 
!   Routines: VecSwap(); VecMAXPY(); VecDestroy(); VecDestroyVecs(); VecDuplicateVecs();
!   Processors: n
!T*/
!
! -----------------------------------------------------------------------

      program ex1f
      implicit none

! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!                    Include files
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!
!  The following include statements are required for Fortran programs
!  that use PETSc vectors:
!     petsc.h  - base PETSc routines
!     vec.h    - vectors
!  Additional include statements may be needed if using additional
!  PETSc routines in a Fortran program, e.g.,
!     viewer.h - viewers
!     is.h     - index sets
!
#include "include/finclude/petsc.h"
#include "include/finclude/vec.h"
!
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!                   Variable declarations
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!
!  Variables:
!     x, y, w - vectors
!     z       - array of vectors

      Vec              x, y, w, z(5)
      double precision norm, v, v1, v2
      integer          n, ierr, flg, rank
      Scalar           one, two, three, dots(3), dot

! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
!                 Beginning of program
! - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

      call PetscInitialize(PETSC_NULL_CHARACTER,ierr)
      one   = 1.0
      two   = 2.0
      three = 3.0
      n     = 20
      call OptionsGetInt(PETSC_NULL_CHARACTER,'-n',n,flg,ierr)
      call MPI_Comm_rank(PETSC_COMM_WORLD,rank,ierr)

!  Create a vector, specifying only its global dimension.
!  When using VecCreate(), the vector format (currently parallel
!  or sequential) is determined at runtime.  Also, the parallel
!  partitioning of the vector is determined by PETSc at runtime.
!
!  Routines for creating particular vector types directly are:
!     VecCreateSeq() - uniprocessor vector
!     VecCreateMPI() - distributed vector, where the user can
!                      determine the parallel partitioning
!     VecCreateShared() - parallel vector that uses shared memory
!                         (available only on the SGI); otherwise,
!                         is the same as VecCreateMPI()
!
!     VecCreate() allows one to determine at runtime which version to use
!                 with the options -vec_mpi or -vec_shared
!
      call VecCreate(PETSC_COMM_WORLD,PETSC_DECIDE,n,x,ierr)

!  Duplicate some work vectors (of the same format and
!  partitioning as the initial vector).

      call VecDuplicate(x,y,ierr)
      call VecDuplicate(x,w,ierr)

!  Duplicate more work vectors (of the same format and
!  partitioning as the initial vector).  Here we duplicate
!  an array of vectors, which is often more convenient than
!  duplicating individual ones.

      call VecDuplicateVecs(x,3,z,ierr)

!  Set the vectors to entries to a constant value.

      call VecSet(one,x,ierr)
      call VecSet(two,y,ierr)
      call VecSet(one,z(1),ierr)
      call VecSet(two,z(2),ierr)
      call VecSet(three,z(3),ierr)

!  Demonstrate various basic vector routines.

      call VecDot(x,x,dot,ierr)
      call VecMDot(3,x,z,dots,ierr)

!  Note: If using a complex numbers version of PETSc, then
!  USE_PETSC_COMPLEX is defined in the makefiles; otherwise,
!  (when using real numbers) it is undefined.

      if (rank .eq. 0) then
#if defined(USE_PETSC_COMPLEX)
         write(6,100) int(PetscReal(dot))
         write(6,110) int(PetscReal(dots(1))),int(PetscReal(dots(2))),             &
     &                int(PetscReal(dots(3)))
#else
         write(6,100) int(dot)
         write(6,110) int(dots(1)),int(dots(2)),int(dots(3))
#endif
         write(6,120)
      endif
 100  format ("Vector length ",i6)
 110  format ("Vector length ",3(i6))
 120  format ("All other values should be near zero")

      call VecScale(two,x,ierr)
      call VecNorm(x,NORM_2,norm,ierr)
      v = norm-2.0*sqrt(dble(n))
      if (v .gt. -1.d-10 .and. v .lt. 1.d-10) v = 0.0
      if (rank .eq. 0) write(6,130) v
 130  format ("VecScale ",e8.2)

      call VecCopy(x,w,ierr)
      call VecNorm(w,NORM_2,norm,ierr)
      v = norm-2.0*sqrt(dble(n))
      if (v .gt. -1.d-10 .and. v .lt. 1.d-10) v = 0.0
      if (rank .eq. 0) write(6,140) v
 140  format ("VecCopy ",e8.2)

      call VecAXPY(three,x,y,ierr)
      call VecNorm(y,NORM_2,norm,ierr)
      v = norm-8.0*sqrt(dble(n))
      if (v .gt. -1.d-10 .and. v .lt. 1.d-10) v = 0.0
      if (rank .eq. 0) write(6,150) v
 150  format ("VecAXPY ",e8.2)

      call VecAYPX(two,x,y,ierr)
      call VecNorm(y,NORM_2,norm,ierr)
      v = norm-18.0*sqrt(dble(n))
      if (v .gt. -1.d-10 .and. v .lt. 1.d-10) v = 0.0
      if (rank .eq. 0) write(6,160) v
 160  format ("VecAYXP ",e8.2)

      call VecSwap(x,y,ierr)
      call VecNorm(y,NORM_2,norm,ierr)
      v = norm-2.0*sqrt(dble(n))
      if (v .gt. -1.d-10 .and. v .lt. 1.d-10) v = 0.0
      if (rank .eq. 0) write(6,170) v
 170  format ("VecSwap ",e8.2)

      call VecNorm(x,NORM_2,norm,ierr)
      v = norm-18.0*sqrt(dble(n))
      if (v .gt. -1.d-10 .and. v .lt. 1.d-10) v = 0.0
      if (rank .eq. 0) write(6,180) v
 180  format ("VecSwap ",e8.2)

      call VecWAXPY(two,x,y,w,ierr)
      call VecNorm(w,NORM_2,norm,ierr)
      v = norm-38.0*sqrt(dble(n))
      if (v .gt. -1.d-10 .and. v .lt. 1.d-10) v = 0.0
      if (rank .eq. 0) write(6,190) v
 190  format ("VecWAXPY ",e8.2)

      call VecPointwiseMult(y,x,w,ierr)
      call VecNorm(w,NORM_2,norm,ierr)
      v = norm-36.0*sqrt(dble(n))
      if (v .gt. -1.d-10 .and. v .lt. 1.d-10) v = 0.0
      if (rank .eq. 0) write(6,200) v
 200  format ("VecPointwiseMult ",e8.2)

      call VecPointwiseDivide(x,y,w,ierr)
      call VecNorm(w,NORM_2,norm,ierr)
      v = norm-9.0*sqrt(dble(n))
      if (v .gt. -1.d-10 .and. v .lt. 1.d-10) v = 0.0
      if (rank .eq. 0) write(6,210) v
 210  format ("VecPointwiseDivide ",e8.2)

      
      dots(1) = one
      dots(2) = three
      dots(3) = two
      call VecSet(one,x,ierr)
      call VecMAXPY(3,dots,x,z,ierr)
      call VecNorm(z(1),NORM_2,norm,ierr)
      v = norm-sqrt(dble(n))
      if (v .gt. -1.d-10 .and. v .lt. 1.d-10) v = 0.0
      call VecNorm(z(2),NORM_2,norm,ierr)
      v1 = norm-2.0*sqrt(dble(n))
      if (v1 .gt. -1.d-10 .and. v1 .lt. 1.d-10) v1 = 0.0
      call VecNorm(z(3),NORM_2,norm,ierr)
      v2 = norm-3.0*sqrt(dble(n))
      if (v2 .gt. -1.d-10 .and. v2 .lt. 1.d-10) v2 = 0.0
      if (rank .eq. 0) write(6,220) v,v1,v2
 220  format ("VecMAXPY ",3(e8.2))


!  Test whether vector has been corrupted (just to demonstrate this
!  routine) not needed in most application codes.

      call VecValid(x,flg,ierr)
      if (flg .ne. PETSC_TRUE) then
         if (rank .eq. 0) then
            write(6,*) 'Corrupted vector!'
         endif
         SETERRA(1,0,' ')
      endif

!  Free work space.  All PETSc objects should be destroyed when they
!  are no longer needed.

      call VecDestroy(x,ierr)
      call VecDestroy(y,ierr)
      call VecDestroy(w,ierr)
      call VecDestroyVecs(z,3,ierr)
      call PetscFinalize(ierr)

      end
 
