      subroutine dpjbfg(nx,ny,x,f,fgrad,task,ecc,b)
      character*(*) task
      integer nx, ny
      double precision f, ecc, b
      double precision x(nx*ny), fgrad(nx*ny)
!     **********
!
!     Subroutine dpjbfg
!
!     This subroutine computes the function and gradient of the
!     pressure distribution in a journal bearing problem.
!
!     The subroutine statement is
!
!       subroutine dpjbfg(nx,ny,x,f,fgrad,task,ecc,b)
!
!     where
!
!       nx is an integer variable.
!         On entry nx is the number of grid points in the first
!            coordinate direction.
!         On exit nx is unchanged.
!
!       ny is an integer variable.
!         On entry ny is the number of grid points in the second
!            coordinate direction.
!         On exit ny is unchanged.
!
!       x is a double precision array of dimension nx*ny.
!         On entry x specifies the vector x if task = 'F', 'G', or 'FG'.
!            Otherwise x need not be specified.
!         On exit x is unchanged if task = 'F', 'G', or 'FG'. Otherwise
!            x is set according to task.
!
!       f is a double precision variable.
!         On entry f need not be specified.
!         On exit f is set to the function evaluated at x if task = 'F'
!            or 'FG'.
!
!       fgrad is a double precision array of dimension nx*ny.
!         On entry fgrad need not be specified.
!         On exit fgrad contains the gradient evaluated at x if
!            task = 'G' or 'FG'.
!
!       task is a character variable.
!         On entry task specifies the action of the subroutine:
!
!            task               action
!            ----               ------
!             'F'     Evaluate the function at x.
!             'G'     Evaluate the gradient at x.
!             'FG'    Evaluate the function and the gradient at x.
!             'XS'    Set x to the standard starting point xs.
!             'XL'    Set x to the lower bound xl.
!
!         On exit task is unchanged.
!
!       ecc is a double precision variable
!         On entry ecc is the eccentricity in (0,1).
!         On exit ecc is unchanged
!
!       b is a double precision variable
!         On entry b defines the domain as D = (0,2*pi) X (0,2*b).
!         On exit b is unchanged.
!
!     MINPACK-2 Project. November 1993.
!     Argonne National Laboratory and University of Minnesota.
!     Brett M. Averick and Jorge J. More'.
!
!     **********
      double precision four, one, p5, six, two, zero
      parameter (zero=0.0d0,p5=0.5d0,one=1.0d0,two=2.0d0,four=4.0d0)
      parameter ( six=6.0d0)

      logical feval, geval
      integer i, j, k
      double precision dvdx, dvdy, ehxhy, flin, fquad, hx, hxhy, hy, pi
      double precision temp, trule, v, vb, vl, vr, vt, xi

      double precision p

      p(xi) = (1+ecc*cos(xi))**3

!     Initialization.

      pi = four*atan(one)
      hx = two*pi/dble(nx+1)
      hy = two*b/dble(ny+1)
      hxhy = hx*hy
      ehxhy = ecc*hxhy

!     Compute the lower bound xl for x if task = 'XL'.

      if (task .eq. 'XL') then
         do 10 k = 1, nx*ny
            x(k) = zero
   10    continue

         return

      end if

!     Compute the standard starting point if task = 'XS'.

      if (task .eq. 'XS') then
         do 30 i = 1, nx
            temp = max(sin(dble(i)*hx),zero)
            do 20 j = 1, ny
               k = nx*(j-1) + i
               x(k) = temp
   20       continue
   30    continue

         return

      end if

      if (task .eq. 'F' .or. task .eq. 'FG') then
         feval = .true.
      else
         feval = .false.
      end if
      if (task .eq. 'G' .or. task .eq. 'FG') then
         geval = .true.
      else
         geval = .false.
      end if

!     Compute the function if task = 'F', the gradient if task = 'G', or
!     both if task = 'FG'.

      if (feval) then
         fquad = zero
         flin = zero
      end if
      if (geval) then
         do 40 k = 1, nx*ny
            fgrad(k) = zero
   40    continue
      end if

!     Computation of the quadratic part of the function and
!     corresponding components of the gradient over the
!     lower triangular elements.

      do 60 i = 0, nx
         xi = dble(i)*hx
         trule = hxhy*(p(xi)+p(xi+hx)+p(xi))/six
         do 50 j = 0, ny
            k = nx*(j-1) + i
            v = zero
            vr = zero
            vt = zero
            if (i .ne. 0 .and. j .ne. 0) v = x(k)
            if (i .ne. nx .and. j .ne. 0) vr = x(k+1)
            if (i .ne. 0 .and. j .ne. ny) vt = x(k+nx)
            dvdx = (vr-v)/hx
            dvdy = (vt-v)/hy
            if (feval) fquad = fquad + trule*(dvdx**2+dvdy**2)
            if (geval) then
               if (i .ne. 0 .and. j .ne. 0)                             &
     &             fgrad(k) = fgrad(k) - trule*(dvdx/hx+dvdy/hy)
               if (i .ne. nx .and. j .ne. 0)                            &
     &             fgrad(k+1) = fgrad(k+1) + trule*dvdx/hx
               if (i .ne. 0 .and. j .ne. ny)                            &
     &             fgrad(k+nx) = fgrad(k+nx) + trule*dvdy/hy
            end if
   50    continue
   60 continue

!     Computation of the quadratic part of the function and
!     corresponding components of the gradient over the upper
!     triangular elements.

      do 80 i = 1, nx + 1
         xi = dble(i)*hx
         trule = hxhy*(p(xi)+p(xi-hx)+p(xi))/six
         do 70 j = 1, ny + 1
            k = nx*(j-1) + i
            vb = zero
            vl = zero
            v = zero
            if (i .ne. nx+1 .and. j .ne. 1) vb = x(k-nx)
            if (i .ne. 1 .and. j .ne. ny+1) vl = x(k-1)
            if (i .ne. nx+1 .and. j .ne. ny+1) v = x(k)
            dvdx = (v-vl)/hx
            dvdy = (v-vb)/hy
            if (feval) fquad = fquad + trule*(dvdx**2+dvdy**2)
            if (geval) then
               if (i .le. nx .and. j .gt. 1)                            &
     &             fgrad(k-nx) = fgrad(k-nx) - trule*dvdy/hy
               if (i .gt. 1 .and. j .le. ny)                            &
     &             fgrad(k-1) = fgrad(k-1) - trule*dvdx/hx
               if (i .le. nx .and. j .le. ny)                           &
     &             fgrad(k) = fgrad(k) + trule*(dvdx/hx+dvdy/hy)
            end if
   70    continue
   80 continue

!     Computation of the linear part of the function and
!     corresponding components of the gradient.

      do 110 i = 1, nx
         temp = sin(dble(i)*hx)
         if (feval) then
            do 90 j = 1, ny
               k = nx*(j-1) + i
               flin = flin + temp*x(k)
   90       continue
         end if
         if (geval) then
            do 100 j = 1, ny
               k = nx*(j-1) + i
               fgrad(k) = fgrad(k) - ehxhy*temp
  100       continue
         end if
  110 continue

!     Finish off the function.

      if (feval) f = p5*fquad - ehxhy*flin

      end
