#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: ex11.c,v 1.10 1998/03/20 22:52:39 bsmith Exp $";
#endif

static char help[] =
"This program demonstrates use of the SNES package to solve systems of\n\
nonlinear equations in parallel, using 2-dimensional distributed arrays.\n\
The 2-dim Bratu (SFI - solid fuel ignition) test problem is used, where\n\
analytic formation of the Jacobian is the default.  \n\
\n\
  Solves the linear systems via 2 level additive Schwarz \n\
\n\
The command line\n\
options are:\n\
  -par <parameter>, where <parameter> indicates the problem's nonlinearity\n\
     problem SFI:  <parameter> = Bratu parameter (0 <= par <= 6.81)\n\
  -mx <xg>, where <xg> = number of grid points in the x-direction\n\
  -my <yg>, where <yg> = number of grid points in the y-direction\n\
  -Nx <npx>, where <npx> = number of processors in the x-direction\n\
  -Ny <npy>, where <npy> = number of processors in the y-direction\n\n";

/*  
    1) Solid Fuel Ignition (SFI) problem.  This problem is modeled by
    the partial differential equation
  
            -Laplacian u - lambda*exp(u) = 0,  0 < x,y < 1 ,
  
    with boundary conditions
   
             u = 0  for  x = 0, x = 1, y = 0, y = 1.
  
    A finite difference approximation with the usual 5-point stencil
    is used to discretize the boundary value problem to obtain a nonlinear 
    system of equations.
*/

#include "snes.h"
#include "da.h"
#include "mg.h"
#include <math.h>

/* User-defined application context */
typedef struct {
   double      param;           /* test problem parameter */
   int         mx,my;           /* fine discretization in x, y directions */
   int         Mx,My;           /* coarse discretization in x, y directions */
   Vec         localX,localF;   /* ghosted local vector */
   DA          da;              /* distributed array data structure */
   Mat         J_coarse,J_fine; /* Jacobians on the two grids */
   SLES        sles_coarse;
   int         ratio;
} AppCtx;

#define COARSE_LEVEL 0
#define FINE_LEVEL   1

int FormFunction(SNES,Vec,Vec,void*), FormInitialGuess1(AppCtx*,Vec);
int FormJacobian(SNES,Vec,Mat*,Mat*,MatStructure*,void*);
int FormInterpolation(Appctx,Mat*);

/*
      Mm_ratio - ration of grid lines between fine and coarse grids.
*/
int main( int argc, char **argv )
{
  SNES          snes;                      
  Vec           x, r, x_coarse, r_coarse;                      
  AppCtx        user;                      
  int           ierr, its, N, n, Nx = PETSC_DECIDE, Ny = PETSC_DECIDE;
  int           size, flg, user.ratio = 2; 
  double        bratu_lambda_max = 6.81, bratu_lambda_min = 0.;
  SLES          sles,sles_fine;
  PC            pc;

  PetscInitialize( &argc, &argv,(char *)0,help );

  user.Mx = 2; user.My = 2; user.param = 6.0;
  ierr = OptionsGetInt(PETSC_NULL,"-Mx",&user.Mx,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-My",&user.My,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-ratio",&user.ratio,&flg); CHKERRA(ierr);
  user.mx = user.ratio*(user.Mx-1)+1; user.my = user.ratio*(user.My-1)+1;

  ierr = OptionsGetDouble(PETSC_NULL,"-par",&user.param,&flg); CHKERRA(ierr);
  if (user.param >= bratu_lambda_max || user.param <= bratu_lambda_min) {
    SETERRA(1,0,"Lambda is out of range");
  }
  n = user.mx*user.my; N = user.Mx*user.My;

  MPI_Comm_size(PETSC_COMM_WORLD,&size);
  ierr = OptionsGetInt(PETSC_NULL,"-Nx",&Nx,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-Ny",&Ny,&flg); CHKERRA(ierr);
  if (Nx*Ny != size && (Nx != PETSC_DECIDE || Ny != PETSC_DECIDE))
    SETERRQ(1,0,"Incompatible number of processors:  Nx * Ny != size");
 
  /* Set up distributed array */
  ierr = DACreate2d(PETSC_COMM_WORLD,DA_NONPERIODIC,DA_STENCIL_STAR,user.mx,
                    user.my,Nx,Ny,1,1,PETSC_NULL,PETSC_NULL,&user.da); CHKERRA(ierr);
  ierr = DACreateLocalVector(user.da,&x); CHKERRA(ierr);
  ierr = VecDuplicate(x,&r); CHKERRA(ierr);
  ierr = DACreateLocalVector(user.da,&user.localX); CHKERRA(ierr);
  ierr = VecDuplicate(user.localX,&user.localF); CHKERRA(ierr);

  /* Create nonlinear solver */
  ierr = SNESCreate(PETSC_COMM_WORLD,SNES_NONLINEAR_EQUATIONS,&snes);CHKERRA(ierr);

  /* provide user function and Jacobian */
  ierr = SNESSetFunction(snes,r,FormFunction,(void *)&user); CHKERRA(ierr);
  ierr = MatCreate(PETSC_COMM_WORLD,n,n,&user.J_fine); CHKERRA(ierr);
  ierr = SNESSetJacobian(snes,user.J_fine,user.J_fine,FormJacobian,&user);CHKERRA(ierr);

  /* set two level additive Schwarz preconditioner */
  ierr = SNESGetSLES(snes,&sles);CHKERRA(ierr);
  ierr = SLESGetPC(sles,&pc); CHKERRA(ierr);
  ierr = PCSetType(pc,PCMG); CHKERRA(ierr);
  ierr = MGSetLevels(pc,2); CHKERRA(ierr);
  ierr = MGSetType(pc,MGADDITIVE); CHKERRA(ierr);

  /* Create coarse level */
  ierr = MGGetCoarseSolve(pc,&user.sles_coarse); CHKERRA(ierr);
  ierr = SLESSetOptionsPrefix(user.sles_coarse,"coarse_"); CHKERRA(ierr);
  ierr = SLESSetFromOptions(user.sles_coarse); CHKERRA(ierr);
  ierr = MatCreate(PETSC_COMM_SELF,N,N,&user.J_coarse); CHKERRA(ierr);
  ierr = SLESSetOperators(user.sles_coarse,user.J_coarse,user.J_coarse,DIFFERENT_NONZERO_PATTERN);
         CHKERRA(ierr);
  ierr = VecCreateSeq(PETSC_COMM_SELF,N,&x_coarse); CHKERRA(ierr);
  ierr = VecDuplicate(x_coarse,&r_coarse); CHKERRA(ierr);
  ierr = MGSetX(pc,COARSE_LEVEL,x_coarse);CHKERRA(ierr); 
  ierr = MGSetRhs(pc,COARSE_LEVEL,r_coarse);CHKERRA(ierr); 

  /* Create fine level */
  ierr = MGGetSmoother(pc,FINE_LEVEL,&sles_fine); CHKERRA(ierr);
  ierr = SLESSetOptionsPrefix(sles_fine,"fine_"); CHKERRA(ierr);
  ierr = SLESSetFromOptions(sles_fine); CHKERRA(ierr);
  ierr = SLESSetOperators(sles_fine,user.J_fine,user.J_fine,DIFFERENT_NONZERO_PATTERN);
         CHKERRA(ierr);

  /* Set options, then solve nonlinear system */
  ierr = SNESSetFromOptions(snes); CHKERRA(ierr);
  ierr = FormInitialGuess1(&user,x); CHKERRA(ierr);
  ierr = SNESSolve(snes,x,&its); CHKERRA(ierr);
  PetscPrintf(PETSC_COMM_WORLD,"Number of Newton iterations = %d\n", its );

  /* Free data structures */
  ierr = MatDestroy(user.J_fine); CHKERRA(ierr);
  ierr = MatDestroy(user.J_coarse); CHKERRA(ierr);
  ierr = VecDestroy(x); CHKERRA(ierr);
  ierr = VecDestroy(r); CHKERRA(ierr);
  ierr = VecDestroy(user.localX); CHKERRA(ierr);
  ierr = VecDestroy(user.localF); CHKERRA(ierr);
  ierr = SNESDestroy(snes); CHKERRA(ierr);
  ierr = DADestroy(user.da); CHKERRA(ierr);
  PetscFinalize();

  return 0;
}/* --------------------  Form initial approximation ----------------- */
int FormInitialGuess1(AppCtx *user,Vec X)
{
  int     i, j, row, mx, my, ierr, xs, ys, xm, ym, Xm, Ym, Xs, Ys;
  double  one = 1.0, lambda, temp1, temp, hx, hy, hxdhy, hydhx,sc;
  Scalar  *x;
  Vec     localX = user->localX;

  mx = user->mx;            my = user->my;            lambda = user->param;
  hx = one/(double)(mx-1);  hy = one/(double)(my-1);
  sc = hx*hy*lambda;        hxdhy = hx/hy;            hydhx = hy/hx;

  /* Get ghost points */
  ierr = VecGetArray(localX,&x); CHKERRQ(ierr);
  temp1 = lambda/(lambda + one);
  ierr = DAGetCorners(user->da,&xs,&ys,0,&xm,&ym,0); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(user->da,&Xs,&Ys,0,&Xm,&Ym,0); CHKERRQ(ierr);

  /* Compute initial guess */
  for (j=ys; j<ys+ym; j++) {
    temp = (double)(PetscMin(j,my-j-1))*hy;
    for (i=xs; i<xs+xm; i++) {
      row = i - Xs + (j - Ys)*Xm; 
      if (i == 0 || j == 0 || i == mx-1 || j == my-1 ) {
        x[row] = 0.0; 
        continue;
      }
      x[row] = temp1*sqrt( PetscMin( (double)(PetscMin(i,mx-i-1))*hx,temp) ); 
    }
  }
  ierr = VecRestoreArray(localX,&x); CHKERRQ(ierr);

  /* Insert values into global vector */
  ierr = DALocalToGlobal(user->da,localX,INSERT_VALUES,X); CHKERRQ(ierr);
  return 0;
} /* --------------------  Evaluate Function F(x) --------------------- */
int FormFunction(SNES snes,Vec X,Vec F,void *ptr)
{
  AppCtx  *user = (AppCtx *) ptr;
  int     ierr, i, j, row, mx, my, xs, ys, xm, ym, Xs, Ys, Xm, Ym;
  double  two = 2.0, one = 1.0, lambda,hx, hy, hxdhy, hydhx,sc;
  Scalar  u, uxx, uyy, *x,*f;
  Vec     localX = user->localX, localF = user->localF; 

  mx = user->mx;            my = user->my;            lambda = user->param;
  hx = one/(double)(mx-1);  hy = one/(double)(my-1);
  sc = hx*hy*lambda;        hxdhy = hx/hy;            hydhx = hy/hx;

  /* Get ghost points */
  ierr = DAGlobalToLocalBegin(user->da,X,INSERT_VALUES,localX); CHKERRQ(ierr);
  ierr = DAGlobalToLocalEnd(user->da,X,INSERT_VALUES,localX); CHKERRQ(ierr);
  ierr = VecGetArray(localX,&x); CHKERRQ(ierr);
  ierr = VecGetArray(localF,&f); CHKERRQ(ierr);
  ierr = DAGetCorners(user->da,&xs,&ys,0,&xm,&ym,0); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(user->da,&Xs,&Ys,0,&Xm,&Ym,0); CHKERRQ(ierr);

  /* Evaluate function */
  for (j=ys; j<ys+ym; j++) {
    row = (j - Ys)*Xm + xs - Xs - 1; 
    for (i=xs; i<xs+xm; i++) {
      row++;
      if (i == 0 || j == 0 || i == mx-1 || j == my-1 ) {
        f[row] = x[row];
        continue;
      }
      u = x[row];
      uxx = (two*u - x[row-1] - x[row+1])*hydhx;
      uyy = (two*u - x[row-Xm] - x[row+Xm])*hxdhy;
      f[row] = uxx + uyy - sc*exp(u);
    }
  }
  ierr = VecRestoreArray(localX,&x); CHKERRQ(ierr);
  ierr = VecRestoreArray(localF,&f); CHKERRQ(ierr);

  /* Insert values into global vector */
  ierr = DALocalToGlobal(user->da,localF,INSERT_VALUES,F); CHKERRQ(ierr);
  PLogFlops(11*ym*xm);
  return 0; 
} 

int FormJacobian_fine(AppCtx *user,Vec X, Mat *J,Mat *B,MatStructure *flag)
{
  Mat     jac = *J;
  int     ierr, i, j, row, mx, my, xs, ys, xm, ym, Xs, Ys, Xm, Ym, col[5];
  int     nloc, *ltog, grow;
  Scalar  two = 2.0, one = 1.0, lambda, v[5], hx, hy, hxdhy, hydhx, sc, *x;
  Vec     localX = user->localX;

  mx = user->mx;            my = user->my;            lambda = user->param;
  hx = one/(double)(mx-1);  hy = one/(double)(my-1);
  sc = hx*hy;               hxdhy = hx/hy;            hydhx = hy/hx;

  /* Get ghost points */
  ierr = DAGlobalToLocalBegin(user->da,X,INSERT_VALUES,localX); CHKERRQ(ierr);
  ierr = DAGlobalToLocalEnd(user->da,X,INSERT_VALUES,localX); CHKERRQ(ierr);
  ierr = VecGetArray(localX,&x); CHKERRQ(ierr);
  ierr = DAGetCorners(user->da,&xs,&ys,0,&xm,&ym,0); CHKERRQ(ierr);
  ierr = DAGetGhostCorners(user->da,&Xs,&Ys,0,&Xm,&Ym,0); CHKERRQ(ierr);
  ierr = DAGetGlobalIndices(user->da,&nloc,&ltog); CHKERRQ(ierr);

  /* Evaluate function */
  for (j=ys; j<ys+ym; j++) {
    row = (j - Ys)*Xm + xs - Xs - 1; 
    for (i=xs; i<xs+xm; i++) {
      row++;
      grow = ltog[row];
      if (i == 0 || j == 0 || i == mx-1 || j == my-1 ) {
        ierr = MatSetValues(jac,1,&grow,1,&grow,&one,INSERT_VALUES); CHKERRQ(ierr);
        continue;
      }
      v[0] = -hxdhy; col[0] = ltog[row - Xm];
      v[1] = -hydhx; col[1] = ltog[row - 1];
      v[2] = two*(hydhx + hxdhy) - sc*lambda*exp(x[row]); col[2] = grow;
      v[3] = -hydhx; col[3] = ltog[row + 1];
      v[4] = -hxdhy; col[4] = ltog[row + Xm];
      ierr = MatSetValues(jac,1,&grow,5,col,v,INSERT_VALUES); CHKERRQ(ierr);
    }
  }
  ierr = MatAssemblyBegin(jac,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  ierr = VecRestoreArray(X,&x); CHKERRQ(ierr);
  ierr = MatAssemblyEnd(jac,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  *flag = SAME_NONZERO_PATTERN;

  return 0;
}

int FormJacobian_coarse(AppCtx *user)
{
  int     ierr, i, j, row, Mx, My, xs, ys, xm, ym, Xs, Ys, Xm, Ym, col[5];
  int     nloc, *ltog, grow;
  Scalar  two = 2.0, one = 1.0, lambda, v[5], hx, hy, hxdhy, hydhx, sc, *x;
  Mat     J = user->J_coarse;

  Mx = user->Mx;            My = user->My;            lambda = user->param;
  hx = one/(double)(Mx-1);  hy = one/(double)(My-1);
  sc = hx*hy;               hxdhy = hx/hy;            hydhx = hy/hx;

  ierr = MatAssemblyBegin(J,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  ierr = MatAssemblyEnd(J,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  ierr = SLESSetOperators(user->sles_coarse,J,J,SAME_NONZERO_PATTERN);

  return 0;
}

/* --------------------  Evaluate Jacobian F'(x) --------------------- */
int FormJacobian(SNES snes,Vec X,Mat *J,Mat *B,MatStructure *flag,void *ptr)
{
  AppCtx  *user = (AppCtx *) ptr;
  int     ierr;

  ierr = FormJacobian_fine(user,X,J,B,flag); CHKERRQ(ierr);

  /* create coarse grid jacobian for preconditioner */
  ierr = FormJacobian_coarse(user); CHKERRQ(ierr);

  return 0;
}


/*
      Forms the interpolation (and restriction) operator from 
coarse grid to fine.
*/
int FormInterpolation(AppCtx *user,Mat *I)
{
  int    i,j,i_start,m,j_start,n,Mx = user->Mx,My = user->My,*idx;
  int    row,i_start_ghost,j_start_ghost,cols[4],mx = user->mx, my = user->my;
  int    c0,c1,c2,c3,nc,ratio = user->ratio;
  Scalar v[4],v0,v1,v2,v3;

  ierr = DAGetCorners(user->da,&i_start,&i_end,0,&m,&n,0);CHKERRQ(ierr);
  ierr = DAGetGhostCorners(user->da,&i_start_ghost,&i_end_ghost,0,&m,&n,0);
         CHKERRQ(ierr);
  ierr = DAGetGlobalIndices(user->da,PETSC_NULL,&idx); CHKERRQ(ierr);
  ierr = MatCreateSeqAIJ(PETSC_COMM_SELF,m*n,Mx*My,PETSC_NULL,PETSC_NULL,I);
         CHKERRQ(ierr);

  /* loop over local fine grid nodes setting interpolation for those*/
  for ( j=j_start; j<j_start+n; j++ ) {
    for ( i=i_start; i<i_start+m; i++ ) {
      row    = idx[m*(j-j_start_ghost) + (i-i_start_ghost)];

      c0 = My*(j/ratio) + (i/ratio);
      /* 
         Only include those interpolation points that are truly 
         nonzero. Note this is very important for final grid lines
         in x and y directions; since they have no right/top neighbors
      */
      nc = 0;
      if (v0) { v[nc] = v0; cols[nc++] = c0; }
      if (v1) { v[nc] = v1; cols[nc++] = c0 + 1; }
      if (v2) { v[nc] = v2; cols[nc++] = c0 + Mx; }
      if (v3) { v[nc] = v3; cols[nc++] = c0 + Mx + 1; }
      ierr = MatSetValues(*I,1,&row,nc,cols,v,INSERT_VALUES); CHKERRQ(ierr);
    }
  }

  ierr = MatAssemblyBegin(*I,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  ierr = MatAssemblyEnd(*I,MAT_FINAL_ASSEMBLY); CHKERRQ(ierr);
  return 0;
}




