#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: gvecksp.c,v 1.17 1998/04/27 14:20:45 curfman Exp $";
#endif

#include "sles.h"
#include "src/gvec/gvecimpl.h"    /*I "gvec.h" I*/


#undef __FUNC__  
#define __FUNC__ "GVecSolutionKSPMonitor"
/*@
   GVecSolutionKSPMonitor - Monitors solution at each KSP iteration.

   Collective on KSP

   Input Parameters:
+  ksp - the Krylov subspace context
.  it  - the number of iterations so far
.  rnorm - the current (approximate) residual norm
-  ctx - viewer

.seealso: KSPDefaultMonitor(),KSPSetMonitor(),GVecResidualKSPMonitor(),
          GVecRhsKSPMonitor(), GVecErrorKSPMonitor()
@*/
int GVecSolutionKSPMonitor(KSP ksp,int it,double rnorm,void* dummy)
{
  Vec    x;
  Viewer ctx = (Viewer)dummy;
  int    ierr;

  PetscFunctionBegin;
  ierr = KSPBuildSolution(ksp,PETSC_NULL,&x); CHKERRQ(ierr);
  ierr = GVecView(x,ctx); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecResidualKSPMonitor" 
/*@
   GVecResidualKSPMonitor - Monitors residual at each KSP iteration.

   Collective on KSP

   Input Parameters:
+  ksp - the Krylov subspace context
.  it  - the number of iterations so far
.  rnorm - the current (approximate) residual norm
-  ctx - viewer

.seealso: KSPDefaultMonitor(),KSPSetMonitor(),GVecSolutionKSPMonitor(),
          GVecRhsKSPMonitor(), GVecErrorKSPMonitor()
@*/
int GVecResidualKSPMonitor(KSP ksp,int it,double rnorm,void* dummy)
{
  Vec    x;
  Viewer ctx = (Viewer)dummy;
  int    ierr;

  PetscFunctionBegin;
  ierr = KSPBuildResidual(ksp,PETSC_NULL,PETSC_NULL,&x); CHKERRQ(ierr);
  ierr = GVecView(x,ctx); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecRhsKSPMonitor" 
/*@
   GVecRhsKSPMonitor - Displays the right hand side for the linear system
   at the first iteration.

   Collective on KSP

   Input Parameters:
+  ksp - the Krylov subspace context
.  it  - the number of iterations so far
.  rnorm - the current (approximate) residual norm
-  ctx - viewer

.seealso: KSPDefaultMonitor(),KSPSetMonitor(),GVecSolutionKSPMonitor(),
          GVecResidualKSPMonitor(), GVecErrorKSPMonitor()
@*/
int GVecRhsKSPMonitor(KSP ksp,int it,double rnorm,void* dummy)
{
  Vec    x;
  Viewer ctx = (Viewer)dummy;
  int    ierr;

  PetscFunctionBegin;
  if (!it) {
    ierr = KSPGetRhs(ksp,&x); CHKERRQ(ierr);
    ierr = GVecView(x,ctx); CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecErrorKSPMonitor" 
/*@
   GVecErrorKSPMonitor - Displays the error at each iteration.

   Collective on KSP

   Input Parameters:
+  ksp - the Krylov subspace context
.  it  - the number of iterations so far
.  rnorm - the current (approximate) residual norm
-  ctx - viewer

   Notes: 
   The final argument to KSPSetMonitor() with this routine must be
   a pointer to a GVecErrorKSPMonitorCtx.

.seealso: KSPDefaultMonitor(),KSPSetMonitor(),,GVecSolutionKSPMonitor(),
          GVecResidualKSPMonitor(), GVecRhsKSPMonitor()
@*/
int GVecErrorKSPMonitor(KSP ksp,int it,double rnorm,void* dummy)
{
  GVec                   x,e;
  GVecErrorKSPMonitorCtx *ctx = (GVecErrorKSPMonitorCtx *)dummy;
  int                    ierr;
  Scalar                 mone = -1.0;
  double                 norm_2,norm_max;
  FILE                   *file;
  MPI_Comm               comm;

  PetscFunctionBegin;
  ierr = KSPBuildSolution(ksp,PETSC_NULL,&x); CHKERRQ(ierr);
  ierr = GVecGetWorkGVec(ctx->solution,&e); CHKERRQ(ierr);
  ierr = VecWAXPY(&mone,x,ctx->solution,e); CHKERRQ(ierr);
  ierr = GVecView(e,ctx->error_viewer); CHKERRQ(ierr);

  /*
      Compute 2-norm and max-norm of error
  */
  if (ctx->norm_error_viewer) {
    ierr = PetscObjectGetComm((PetscObject)ksp,&comm); CHKERRQ(ierr);
    ierr = ViewerASCIIGetPointer(ctx->norm_error_viewer,&file); CHKERRQ(ierr);
    ierr = VecNorm(e,NORM_2,&norm_2);CHKERRQ(ierr);
    ierr = VecNorm(e,NORM_MAX,&norm_max);CHKERRQ(ierr);
    PetscFPrintf(comm,file,"Iteration %d residual norm %g error 2-norm %g error max-norm %g\n",
                 it,rnorm,norm_2,norm_max);
  }

  ierr = GVecRestoreWorkGVec(ctx->solution,&e); CHKERRQ(ierr);  
  PetscFunctionReturn(0);
}

#include "mg.h"
/* ------------------------------------------------------------------------------*/
#undef __FUNC__  
#define __FUNC__ "GVecPCMGSetMonitor" 
/*
    GVecPCMGSetMonitor - Sets GVec KSP monitors for each level of multigrid.

   Input Parameters:
.    pc - preconditioner context for multigrid
*/
int GVecPCMGSetMonitor(PC pc,int views)
{
  char   name[25];
  int    levels, i,ierr, solution,rhs,residual,all;
  SLES   subsles;
  KSP    ksp;
  Viewer *viewers;
  PCType type;

  PetscFunctionBegin;
  PetscValidHeaderSpecific(pc,PC_COOKIE);
  ierr = PCGetType(pc,&type);CHKERRQ(ierr);
  if (PetscStrcmp(type,PCMG)) PetscFunctionReturn(0);

  solution = (views & GVECPCMGMONITOR_SOLUTION);
  residual = (views & GVECPCMGMONITOR_RESIDUAL);
  rhs      = (views & GVECPCMGMONITOR_RHS);
  all      = (solution != 0) + (residual != 0) + (rhs != 0);

  ierr = MGGetLevels(pc,&levels); CHKERRQ(ierr);

  viewers = (Viewer*) PetscMalloc( all*levels*sizeof(Viewer) ); CHKPTRQ(viewers);

  for ( i=0; i<levels; i++ ) {
    ierr = MGGetSmoother(pc,i,&subsles); CHKERRQ(ierr);
    ierr = SLESGetKSP(subsles,&ksp);  CHKERRQ(ierr);
    if (rhs) {
      sprintf(name,"Right hand side %d",i);
      ierr = ViewerDrawOpenX(PETSC_COMM_WORLD,0,name,PETSC_DECIDE,PETSC_DECIDE,200,200,viewers);
             CHKERRQ(ierr);
      ierr = KSPSetMonitor(ksp,GVecRhsKSPMonitor,*viewers++);CHKERRQ(ierr);
    }
    if (solution) {
      sprintf(name,"Solution %d",i);
      ierr = ViewerDrawOpenX(PETSC_COMM_WORLD,0,name,PETSC_DECIDE,PETSC_DECIDE,200,200,viewers);
             CHKERRQ(ierr);
      ierr = KSPSetMonitor(ksp,GVecSolutionKSPMonitor,*viewers++);CHKERRQ(ierr);
    }
    if (residual) {
      sprintf(name,"Residual %d",i);
      ierr = ViewerDrawOpenX(PETSC_COMM_WORLD,0,name,PETSC_DECIDE,PETSC_DECIDE,200,200,viewers);
             CHKERRQ(ierr);
      ierr = KSPSetMonitor(ksp,GVecResidualKSPMonitor,*viewers++);CHKERRQ(ierr);
    }
  }

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GVecKSPOptionsChecker_Private" 
int GVecKSPOptionsChecker_Private(KSP ksp) 
{
  char     *prefix,string[64];  
  int      ierr,flg,loc[4],nmax;
  MPI_Comm comm;
  Viewer   viewer;
  Draw     draw;

  PetscFunctionBegin;
  ierr = KSPGetOptionsPrefix(ksp,&prefix); CHKERRQ(ierr);
  ierr = PetscObjectGetComm((PetscObject)ksp,&comm); CHKERRQ(ierr);

  nmax = 4;
  loc[0] = PETSC_DECIDE; loc[1] = PETSC_DECIDE; loc[2] = 300; loc[3] = 300;
  ierr = OptionsGetIntArray(prefix,"-gvec_ksp_solutionmonitor",loc,&nmax,&flg); CHKERRQ(ierr);
  if (flg) {
    ierr = ViewerDrawOpenX(comm,0,0,loc[0],loc[1],loc[2],loc[3],&viewer); CHKERRQ(ierr);
    ierr = ViewerDrawGetDraw(viewer,&draw); CHKERRQ(ierr);
    ierr = DrawSetTitle(draw,"Approx. Solution"); CHKERRQ(ierr);
    PLogObjectParent(ksp,(PetscObject) viewer);
    ierr = KSPSetMonitor(ksp,GVecSolutionKSPMonitor,(void *)viewer); CHKERRQ(ierr);
  }
  nmax = 4;
  loc[0] = PETSC_DECIDE; loc[1] = PETSC_DECIDE; loc[2] = 300; loc[3] = 300;
  ierr = OptionsGetIntArray(prefix,"-gvec_ksp_residualmonitor",loc,&nmax,&flg); CHKERRQ(ierr);
  if (flg) {
    ierr = ViewerDrawOpenX(comm,0,0,loc[0],loc[1],loc[2],loc[3],&viewer); CHKERRQ(ierr);
    ierr = ViewerDrawGetDraw(viewer,&draw); CHKERRQ(ierr);
    ierr = DrawSetTitle(draw,"Residual"); CHKERRQ(ierr);
    PLogObjectParent(ksp,(PetscObject) viewer);
    ierr = KSPSetMonitor(ksp,GVecResidualKSPMonitor,(void *)viewer); CHKERRQ(ierr);
  }
  nmax = 4;
  loc[0] = PETSC_DECIDE; loc[1] = PETSC_DECIDE; loc[2] = 300; loc[3] = 300;
  ierr = OptionsGetIntArray(prefix,"-gvec_ksp_rhsmonitor",loc,&nmax,&flg); CHKERRQ(ierr);
  if (flg) {
    ierr = ViewerDrawOpenX(comm,0,0,loc[0],loc[1],loc[2],loc[3],&viewer); CHKERRQ(ierr);
    ierr = ViewerDrawGetDraw(viewer,&draw); CHKERRQ(ierr);
    ierr = DrawSetTitle(draw,"Rhs"); CHKERRQ(ierr);
    PLogObjectParent(ksp,(PetscObject) viewer);
    ierr = KSPSetMonitor(ksp,GVecRhsKSPMonitor,(void *)viewer); CHKERRQ(ierr);
  }
  ierr = OptionsGetString(prefix,"-gvec_mg_monitor",string,64,&flg); CHKERRQ(ierr);
  if (flg) {
    PC  pc;
    int all = 0;
    if (!PetscStrstr(string,"nosolution")) all |= GVECPCMGMONITOR_SOLUTION;
    if (!PetscStrstr(string,"noresidual")) all |= GVECPCMGMONITOR_RESIDUAL;
    if (!PetscStrstr(string,"norhs"))      all |= GVECPCMGMONITOR_RHS;
    ierr = KSPGetPC(ksp,&pc);  CHKERRQ(ierr);
    ierr = GVecPCMGSetMonitor(pc,all); CHKERRQ(ierr);
  }
  ierr = OptionsHasName(PETSC_NULL,"-help",&flg); CHKERRQ(ierr);
  if (flg) {
    char pprefix[64];
    PetscStrcpy(pprefix,"-");
    if (prefix)  PetscStrcat(pprefix,prefix);
    (*PetscHelpPrintf)(comm," Additional KSP Monitor options for grid vectors\n",pprefix);
    (*PetscHelpPrintf)(comm,"   %sgvec_ksp_solutionmonitor\n",pprefix);
    (*PetscHelpPrintf)(comm,"   %sgvec_ksp_residualmonitor\n",pprefix);
    (*PetscHelpPrintf)(comm,"   %sgvec_ksp_rhsmonitor\n",pprefix);
    (*PetscHelpPrintf)(comm,"   %sgvec_mg_monitor [nosolution,norhs,noresidual]\n",pprefix);
  }
  PetscFunctionReturn(0);
}

#include "snes.h"
extern int GVecSNESOptionsChecker_Private(SNES);
#undef __FUNC__  
#define __FUNC__ "GVecInitialize"
/*@
    GVecInitialize - Initializes PETSc to use all GVec functionality.

    Not Collective

    Notes:
    Currently this routine initialize the GVec monitoring routines 
    for KSP and SNES solvers.
@*/
int GVecInitialize(void) 
{
  static int initialized = 0;
         int ierr;

  PetscFunctionBegin;
  if (initialized) PetscFunctionReturn(0);
  initialized = 1;

  ierr = KSPAddOptionsChecker(GVecKSPOptionsChecker_Private); CHKERRQ(ierr);
  ierr = SNESAddOptionsChecker(GVecSNESOptionsChecker_Private); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

