#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: rect2d.c,v 1.23 1998/04/27 14:05:03 curfman Exp $";
#endif

/*
   Implements 2d rectangular grids.
*/
#include "src/gvec/impls/rectangular/rectimpl.h"         /*I "gvec.h" I*/
#include "pinclude/pviewer.h"


#undef __FUNC__  
#define __FUNC__ "GridDataView_Rectangular_2D_File"
static int GridDataView_Rectangular_2D_File(GridData gridd,Viewer viewer)
{
  FILE                 *fd;
  int                  ierr,i;
  GridData_Rectangular *s = (GridData_Rectangular *) gridd->data;

  PetscFunctionBegin;
  ierr = ViewerASCIIGetPointer(viewer,&fd); CHKERRQ(ierr);
  fprintf(fd,"GridData Object:\n");
  fprintf(fd,"  Rectangular 2D grid with %d by %d points \n",s->m,s->n);
  for (i=0; i<s->m; i++ ) {
    fprintf(fd,"  %d %g\n",i,s->x[i]);
  }
  for (i=0; i<s->n; i++ ) {
    fprintf(fd,"  %d %g\n",i,s->y[i]);
  }

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataView_Rectangular_2D_Draw"
static int GridDataView_Rectangular_2D_Draw(GridData gridd,Viewer v)
{
  Draw                 draw;
  int                  ierr;
  PetscTruth           isnull;
  /* 
  GridData_Rectangular *s = (GridData_Rectangular *) gridd->data;
  int                  i,m = s->m,ierr,n = s->n;
  double               yt1,yt2,y, *x = s->x,w,h;
  */

  PetscFunctionBegin;
  ierr = ViewerDrawGetDraw(v,&draw); CHKERRQ(ierr);
  ierr = DrawIsNull(draw,&isnull); CHKERRQ(ierr); if (isnull) PetscFunctionReturn(0);


  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataView_Rectangular_2D"
static int GridDataView_Rectangular_2D(GridData gridd,Viewer viewer)
{
  int         rank, ierr;
  ViewerType  vtype;

  PetscFunctionBegin;
  MPI_Comm_rank(gridd->comm,&rank); if (rank > 0) PetscFunctionReturn(0);
  ierr = ViewerGetType(viewer,&vtype);CHKERRQ(ierr);
  if (vtype == ASCII_FILE_VIEWER || vtype == ASCII_FILES_VIEWER){
    ierr =  GridDataView_Rectangular_2D_File(gridd,viewer);CHKERRQ(ierr);
  } else if (vtype == DRAW_VIEWER) {
    ierr = GridDataView_Rectangular_2D_Draw(gridd,viewer);CHKERRQ(ierr);
  } else {
    SETERRQ(1,1,"Viewer type not supported by PETSc object");
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataDestroy_Rectangular_2D"
static int GridDataDestroy_Rectangular_2D(GridData g)
{
  GridData_Rectangular *s = (GridData_Rectangular *)g->data;
  Vec                  local,global;
  int                  ierr;

  PetscFunctionBegin;
  if (s->da) {
    ierr = DACreateGlobalVector(s->da,&global); CHKERRQ(ierr);
    ierr = VecDestroy(global);  CHKERRQ(ierr);
    ierr = DACreateLocalVector(s->da,&local); CHKERRQ(ierr);
    ierr = VecDestroy(local);  CHKERRQ(ierr);
    ierr = DADestroy(s->da); CHKERRQ(ierr);
  }

  PetscFree(s->x);
  PetscFree(s->y);
  PetscFree(s);
  PLogObjectDestroy(g);
  PetscHeaderDestroy(g);

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataRefine_Rectangular_2D"
static int GridDataRefine_Rectangular_2D(GridData gridd,int l,GridData *newgrid)
{
  GridData_Rectangular *s = (GridData_Rectangular *)gridd->data;
  int                  m = s->m, M = 2*m - 1, n = s->n, N = 2*n - 1, i,ierr;
  double               *x = s->x, *xnew, *ynew, *y = s->y;

  PetscFunctionBegin;
  if (l != 2) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Only factor 2 refinement supported");
  xnew    = (double *) PetscMalloc( M*sizeof(double) );CHKPTRQ(xnew);
  xnew[0] = x[0];
  for ( i=1; i<m; i++ ) {
    xnew[2*i-1] = x[i-1] + .5*(x[i] - x[i-1]);
    xnew[2*i]   = x[i];
  }
  ynew    = (double *) PetscMalloc( N*sizeof(double) );CHKPTRQ(ynew);
  ynew[0] = y[0];
  for ( i=1; i<m; i++ ) {
    ynew[2*i-1] = y[i-1] + .5*(y[i] - y[i-1]);
    ynew[2*i]   = y[i];
  }
  ierr = GridDataCreateRectangular2D(gridd->comm,M,0.0,0.0,xnew,N,0.0,0.0,ynew,newgrid);
         CHKERRQ(ierr);
  PetscFree(xnew);
  PetscFree(ynew);
  PetscFunctionReturn(0);
}

extern int GridCreateRectangular2DLinear_Private(GridData,Grid *);

#undef __FUNC__  
#define __FUNC__ "GridDataCreateGrid_Rectangular_2D"
static int GridDataCreateGrid_Rectangular_2D(GridData gridd,DiscretizationType type,Grid *grid)
{
  int              ierr;

  PetscFunctionBegin;
  /*
      This should be reorganized with a registration process
  */
  if (type == DISCRETIZATION_LINEAR) {
    ierr = GridCreateRectangular2DLinear_Private(gridd,grid); CHKERRQ(ierr);
  } else {
    SETERRQ(PETSC_ERR_SUP,0,"Do not support that type discretization");
  }
  PetscFunctionReturn(0);
}

static struct _GridDataOps GDOps = {0,
                                    0,
                                    0,
                                    GridDataCreateGrid_Rectangular_2D,
                                    GridDataRefine_Rectangular_2D};

#undef __FUNC__  
#define __FUNC__ "GridDataCreateRectangular2D"
/*@C
    GridDataCreateRectangular2D - Creates a basic two-dimensional grid.

    Collective on MPI_Comm

    Input Parameters:
+   comm - the communicator that shares the grid
.   m - the number of GridData points in the x direction
.   xmin - the first x grid point
.   xmax - the last x grid point
.   xpoints - an array of grid points
.   n - the number of GridData points in the y direction
.   ymin - the first y grid point
.   ymax - the last y grid point
-   ypoints - an array of grid points

    Output Parameter:
.   grid - the grid

.seealso: GridDataCreateRectangular1D()
@*/
int GridDataCreateRectangular2D(MPI_Comm comm,int m,double xmin,double xmax,
                                double *xpoints,int n, double ymin, double ymax, 
                                double *ypoints, GridData *grid)
{
  GridData             g;
  GridData_Rectangular *s;
  int                  i,flag,ierr;
  double               h;

  PetscFunctionBegin;
  if (m < 2) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Arg m < 2");
  if (n < 2) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Arg n < 2");

  PetscHeaderCreate(g,_p_GridData,struct _GridDataOps,GRIDDATA_COOKIE,GRIDDATA_RECTANGULAR_2D,comm,GridDataDestroy,GridDataView);
  PLogObjectCreate(g);
  PLogObjectMemory(g,sizeof(struct _p_GridData)+sizeof(GridData_Rectangular)+(m+n)*sizeof(double));
  s              = PetscNew(GridData_Rectangular); CHKPTRQ(s);
  PetscMemcpy(g->ops,&GDOps,sizeof(struct _GridDataOps));
  g->data        = (void *) s;
  g->ops->destroy     = GridDataDestroy_Rectangular_2D;
  g->ops->view        = GridDataView_Rectangular_2D;
  g->dim         = 2;
  s->x           = (double *) PetscMalloc(m*sizeof(double));CHKPTRQ(s->x);
  s->m           = m;
  s->y           = (double *) PetscMalloc(n*sizeof(double));CHKPTRQ(s->y);
  s->n           = n;
  s->da          = 0;
  if (xpoints) {
    for ( i=0; i<m; i++ ) {
      s->x[i] = xpoints[i];
    }
  } else {
    h = (xmax - xmin)/(m - 1.0);
    s->x[0] = xmin;
    for ( i=1; i<m; i++ ) {
      s->x[i] = h + s->x[i-1];
    }
  }
  if (ypoints) {
    for ( i=0; i<n; i++ ) {
      s->y[i] = ypoints[i];
    }
  } else {
    h = (ymax - ymin)/(n - 1.0);
    s->y[0] = ymin;
    for ( i=1; i<n; i++ ) {
      s->y[i] = h + s->y[i-1];
    }
  }
  *grid = g;

  ierr = OptionsHasName(PETSC_NULL,"-griddata_view",&flag); CHKERRQ(ierr);
  if (flag) {
    ierr = GridDataView(*grid,0);
  }
  ierr = OptionsHasName(PETSC_NULL,"-griddata_view_draw",&flag); CHKERRQ(ierr);
  if (flag) {
    Viewer    viewer;
    Draw      draw;
    ierr = ViewerDrawOpenX(g->comm,0,0,0,0,300,300,&viewer); CHKERRQ(ierr);
    ierr = GridDataView(g,viewer); CHKERRQ(ierr);
    ierr = ViewerFlush(viewer); CHKERRQ(ierr);
    ierr = ViewerDrawGetDraw(viewer,&draw); CHKERRQ(ierr);
    ierr = DrawPause(draw); CHKERRQ(ierr);
    ierr = ViewerDestroy(viewer); CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridCreateRectangular2D"
/*@C
    GridCreateRectangular2D - Creates a two-dimensional rectangular grid object.

    Collective on MPI_Comm

    Input Parameters:
+   comm - MPI communicator
.   m   - number of grid points in x direction (including end points)
.   n   - number of grid points in y direction (including end points)
.   xmin - left grid point
.   xmax - right grid point
.   ymin - bottom grid point
.   ymax - top grid point
.   x - [optional] array of grid points, only given if the grid points are
        unevenly spaced.
.   y - [optional] array of grid points, only given if the grid points are
        unevenly spaced.
-   dtype - type of discretization, for example DISCRETIZATION_LINEAR

    Output Parameter:
.   grid - the grid data structure

.keywords: grid

.seealso: GridCreateRectangular1D()
@*/
int GridCreateRectangular2D(MPI_Comm comm,int m,double xmin,double xmax,double *x,
                            int n,double ymin, double ymax,double *y,
                            DiscretizationType dtype,Grid *grid)
{
  GridData gridd;
  int      ierr;
  
  PetscFunctionBegin;
  ierr = GridDataCreateRectangular2D(comm,m,xmin,xmax,x,n,ymin,ymax,y,&gridd);CHKERRQ(ierr);
  ierr = GridDataCreateGrid(gridd,dtype,grid); CHKERRQ(ierr);
  /* Remove uneeded reference to gridd */
  ierr = GridDataDestroy(gridd); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}



