#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: rect1d.c,v 1.28 1998/04/27 14:05:37 curfman Exp $";
#endif

/*
   Implements 1d rectangular grids.
*/
#include "src/gvec/impls/rectangular/rectimpl.h"         /*I "gvec.h" I*/
#include "pinclude/pviewer.h"


#undef __FUNC__  
#define __FUNC__ "GridDataView_Rectangular_1D_File"
static int GridDataView_Rectangular_1D_File(GridData gridd,Viewer viewer)
{
  FILE                 *fd;
  int                  ierr,i;
  GridData_Rectangular *s = (GridData_Rectangular *) gridd->data;

  PetscFunctionBegin;
  ierr = ViewerASCIIGetPointer(viewer,&fd); CHKERRQ(ierr);
  fprintf(fd,"GridData Object:\n");
  fprintf(fd,"  Rectangular 1D grid with %d points \n",s->m);
  for (i=0; i<s->m; i++ ) {
    fprintf(fd,"  %d %g\n",i,s->x[i]);
  }

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataView_Rectangular_1D_Draw"
static int GridDataView_Rectangular_1D_Draw(GridData gridd,Viewer v)
{
  Draw                 draw;
  PetscTruth           isnull;
  GridData_Rectangular *s = (GridData_Rectangular *) gridd->data;
  int                  i,m = s->m,ierr;
  double               yt1,yt2,y, *x = s->x,w,h;

  PetscFunctionBegin;
  ierr = ViewerDrawGetDraw(v,&draw); CHKERRQ(ierr);
  ierr = DrawIsNull(draw,&isnull); CHKERRQ(ierr); if (isnull) PetscFunctionReturn(0);

  y = .5; /* this is an arbitrary choice; looks ok for default window */
  ierr = DrawLine(draw,x[0],y,x[m-1],y,DRAW_RED); CHKERRQ(ierr);
  ierr = DrawStringGetSize(draw,&w,&h); CHKERRQ(ierr);
  yt1 = .5 - .5*h; yt2 = .5 + .5*h;
  for ( i=0; i<m; i++ ) {
    ierr = DrawLine(draw,x[i],yt1,x[i],yt2,DRAW_BLUE); CHKERRQ(ierr);
  }

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataView_Rectangular_1D"
static int GridDataView_Rectangular_1D(GridData gridd,Viewer viewer)
{
  int         rank, ierr;
  ViewerType  vtype;

  PetscFunctionBegin;
  MPI_Comm_rank(gridd->comm,&rank); if (rank > 0) PetscFunctionReturn(0);
  ierr = ViewerGetType(viewer,&vtype);CHKERRQ(ierr);
  if (vtype == ASCII_FILE_VIEWER || vtype == ASCII_FILES_VIEWER){
    ierr = GridDataView_Rectangular_1D_File(gridd,viewer);CHKERRQ(ierr);
  } else if (vtype == DRAW_VIEWER) {
    ierr = GridDataView_Rectangular_1D_Draw(gridd,viewer);CHKERRQ(ierr);
  } else {
    SETERRQ(1,1,"Viewer type not supported by PETSc object");
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataDestroy_Rectangular_1D"
static int GridDataDestroy_Rectangular_1D(GridData g)
{
  GridData_Rectangular *s = (GridData_Rectangular *)g->data;
  int                  ierr;

  PetscFunctionBegin;
  if (s->da) {
    ierr = DADestroy(s->da); CHKERRQ(ierr);
  }

  PetscFree(s->x);
  PetscFree(s);
  PLogObjectDestroy(g);
  PetscHeaderDestroy(g);

  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridDataRefine_Rectangular_1D"
static int GridDataRefine_Rectangular_1D(GridData gridd,int l,GridData *newgrid)
{
  GridData_Rectangular *s = (GridData_Rectangular *)gridd->data;
  int                  m = s->m, M = 2*m - 1,i,ierr;
  double               *x = s->x, *xnew;

  PetscFunctionBegin;
  if (l != 2) SETERRQ(PETSC_ERR_SUP,0,"Only factor 2 refinement supported");
  xnew    = (double *) PetscMalloc( M*sizeof(double) );CHKPTRQ(xnew);
  xnew[0] = x[0];
  for ( i=1; i<m; i++ ) {
    xnew[2*i-1] = x[i-1] + .5*(x[i] - x[i-1]);
    xnew[2*i]   = x[i];
  }
  ierr = GridDataCreateRectangular1D(gridd->comm,M,0.0,0.0,xnew,newgrid);CHKERRQ(ierr);
  PetscFree(xnew);
  PetscFunctionReturn(0);
}

extern int GridCreateRectangular1DLinear_Private(GridData,Grid *);
extern int GridCreateRectangular1DQuadratic_Private(GridData,Grid *);

#undef __FUNC__  
#define __FUNC__ "GridDataCreateGrid_Rectangular_1D"
static int GridDataCreateGrid_Rectangular_1D(GridData gridd,DiscretizationType type,Grid *grid)
{
  int              ierr;

  PetscFunctionBegin;
  /*
      This should be reorganized with a registration process
  */
  if (type == DISCRETIZATION_LINEAR) {
    ierr = GridCreateRectangular1DLinear_Private(gridd,grid); CHKERRQ(ierr);
  } else if (type == DISCRETIZATION_QUADRATIC) {
    ierr = GridCreateRectangular1DQuadratic_Private(gridd,grid); CHKERRQ(ierr);
  } else {
    SETERRQ(PETSC_ERR_SUP,0,"Do not support that type discretization");
  }
  PetscFunctionReturn(0);
}

static struct _GridDataOps GDOps = {0,
                                    0,
                                    0,
                                    GridDataCreateGrid_Rectangular_1D,
                                    GridDataRefine_Rectangular_1D};

#undef __FUNC__  
#define __FUNC__ "GridDataCreateRectangular1D"
/*@C
    GridDataCreateRectangular1D - Creates a basic one-dimensional grid.

    Collective on MPI_Comm

    Input Parameters:
+   comm - the communicator that shares the grid
.   m - the number of GridData points
.   xmin - the first grid point
.   xmax - the last grid point
-   points - an array of grid points

    Output Parameter:
.   grid - the grid

.seealso: GridDataCreateRectangular2D()
@*/
int GridDataCreateRectangular1D(MPI_Comm comm,int m,double xmin,double xmax,
                                double *points,GridData *grid)
{
  GridData             g;
  GridData_Rectangular *s;
  int                  i,flag,ierr;
  double               h;

  PetscFunctionBegin;
  if (m < 1) SETERRQ(PETSC_ERR_ARG_OUTOFRANGE,0,"Arg m < 1");

  PetscHeaderCreate(g,_p_GridData,struct _GridDataOps,GRIDDATA_COOKIE,GRIDDATA_RECTANGULAR_1D,comm,GridDataDestroy,GridDataView);
  PLogObjectCreate(g);
  PLogObjectMemory(g,sizeof(struct _p_GridData)+sizeof(GridData_Rectangular)+m*sizeof(double));
  s              = PetscNew(GridData_Rectangular); CHKPTRQ(s);
  PetscMemcpy(g->ops,&GDOps,sizeof(struct _GridDataOps));
  g->data        = (void *) s;
  g->ops->destroy= GridDataDestroy_Rectangular_1D;
  g->ops->view   = GridDataView_Rectangular_1D;
  g->dim         = 1;
  s->x           = (double *) PetscMalloc(m*sizeof(double));CHKPTRQ(s->x);
  s->m           = m;
  s->da          = 0;
  if (points) {
    for ( i=0; i<m; i++ ) {
      s->x[i] = points[i];
    }
  } else {
    h = (xmax - xmin)/(m - 1.0);
    s->x[0] = xmin;
    for ( i=1; i<m; i++ ) {
      s->x[i] = h + s->x[i-1];
    }
  }
  *grid = g;

  ierr = OptionsHasName(PETSC_NULL,"-griddata_view",&flag); CHKERRQ(ierr);
  if (flag) {
    ierr = GridDataView(*grid,0);
  }
  ierr = OptionsHasName(PETSC_NULL,"-griddata_view_draw",&flag); CHKERRQ(ierr);
  if (flag) {
    Viewer    viewer;
    Draw      draw;
    ierr = ViewerDrawOpenX(g->comm,0,0,0,0,300,300,&viewer); CHKERRQ(ierr);
    ierr = GridDataView(g,viewer); CHKERRQ(ierr);
    ierr = ViewerFlush(viewer); CHKERRQ(ierr);
    ierr = ViewerDrawGetDraw(viewer,&draw); CHKERRQ(ierr);
    ierr = DrawPause(draw); CHKERRQ(ierr);
    ierr = ViewerDestroy(viewer); CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

#undef __FUNC__  
#define __FUNC__ "GridCreateRectangular1D"
/*@C
    GridCreateRectangular1D - Creates a one-dimensional rectangular grid object.

    Collective on MPI_Comm

    Input Parameters:
+   comm - MPI communicator
.   n   - number of grid points (including end points)
.   xmin - left grid point
.   xmax - right grid point
.   x - [optional] array of grid points, only given if the grid points are
        unevenly spaced.
-   dtype - type of discretization, for example DISCRETIZATION_LINEAR

    Output Parameter:
.   grid - the grid data structure

.keywords: grid

.seealso: GridCreateRectangular2D()
@*/
int GridCreateRectangular1D(MPI_Comm comm,int n,double xmin,double xmax,double *x,
                            DiscretizationType dtype,Grid *grid)
{
  GridData gridd;
  int      ierr;
  
  PetscFunctionBegin;
  ierr = GridDataCreateRectangular1D(comm,n,xmin,xmax,x,&gridd);CHKERRQ(ierr);
  ierr = GridDataCreateGrid(gridd,dtype,grid); CHKERRQ(ierr);
  /* Remove uneeded reference to grid */
  ierr = GridDataDestroy(gridd); CHKERRQ(ierr);
  PetscFunctionReturn(0);
}



