#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: ex3.c,v 1.10 1997/11/03 04:50:59 bsmith Exp $";
#endif

/*T
    Concepts: Grid vectors
    Routines: GridCreateRectangular1D(); GridCreateGVec()
    Routines: ViewerDrawOpenX(); GridDestroy(); 
    Routines: GVecEvaluateFunctionGalerkin();
    Routines: SNESSetFromOptions(); SNESSolve();
    
    Comment: Solves a simple nonlinear PDE in one dimension
T*/

static char help[] = "Solves a simple nonlinear PDE in one dimension with multigrid.\n\n";


#include "snes.h"
#include "gvec.h"

extern int SolutionFunction(int,double *,double *,double *,Scalar *,void*);

extern int FormJacobian(SNES,Vec,Mat*,Mat*,MatStructure*,void*);
extern int FormFunction(SNES,Vec,Vec,void*);

#include "src/gvec/examples/tutorials/common3and4.c"

int main(int argc,char **args)
{
  GVec                    x,b,r,u;             /* discrete grid vectors */
  GMat                    A;                   /* Jacobian for nonlinear solver */
  Grid                    grid;
  int                     ierr, n = 10, flag, its;
  SNES                    snes;                /* nonlinear solver context */
  Scalar                  one = 1.0;
  GVecErrorSNESMonitorCtx mctx;
  Draw                    draw;

  PetscInitialize(&argc,&args,0,help);
  ierr = GVecInitialize(); CHKERRA(ierr);

  ierr = OptionsGetInt(PETSC_NULL,"-n",&n,&flag); CHKERRA(ierr);

  /*
      Construct the grid and the discretization context
  */
  ierr = GridCreateRectangular1D(PETSC_COMM_WORLD,n,0.0,1.0,PETSC_NULL,
                                 DISCRETIZATION_LINEAR,&grid); CHKERRA(ierr);

  /* 
      From the grid create the grid vector
  */
  ierr = GridCreateGVec(grid,&x); CHKERRA(ierr);

  /*
      Create additional vectors to hold the right-hand-side and exact solution
  */
  ierr = VecDuplicate(x,&b); CHKERRA(ierr);
  ierr = VecDuplicate(x,&r); CHKERRA(ierr);

  /*
       Compute the empty Jacobian matrix
  */
  ierr = GridCreateGMat(grid,&A); CHKERRA(ierr);

  /*
       Set the initial guess
  */
  ierr = VecSet(&one,x); CHKERRA(ierr);

  /*
      Create a vector to hold the exact solution
  */
  ierr = VecDuplicate(x,&u); CHKERRA(ierr);
  ierr = GVecEvaluateFunction(u,SolutionFunction,0); CHKERRA(ierr);

  /*
     Create the nonlinear solver context
  */
  ierr = SNESCreate(PETSC_COMM_WORLD,SNES_NONLINEAR_EQUATIONS,&snes); CHKERRA(ierr);

  /*
     Set the function for which we wish to find the zero and its Jacobian
  */
  ierr = SNESSetFunction(snes,b,FormFunction,PETSC_NULL); CHKERRA(ierr);
  ierr = SNESSetJacobian(snes,A,A,FormJacobian,PETSC_NULL); CHKERRA(ierr);

  /*
      Monitor the error at every iteration
  */
  ierr = ViewerDrawGetDraw(VIEWER_DRAWX_WORLD,&draw); CHKERRA(ierr);
  ierr = DrawSetTitle(draw,"Error"); CHKERRA(ierr);
  mctx.error_viewer      = VIEWER_DRAWX_WORLD;
  mctx.norm_error_viewer = VIEWER_STDOUT_WORLD;
  mctx.solution          = u;
  ierr = SNESSetMonitor(snes,GVecErrorSNESMonitor,&mctx); CHKERRA(ierr);

  /*
     Set various runtime options; then solve the nonlinear system
  */
  ierr = SNESSetFromOptions(snes); CHKERRA(ierr);
  ierr = SNESSolve(snes,x,&its); CHKERRA(ierr);

  /*
     Free work space.  All PETSc objects should be destroyed when they
     are no longer needed.
  */
  ierr = VecDestroy(x); CHKERRA(ierr);
  ierr = VecDestroy(b); CHKERRA(ierr);
  ierr = VecDestroy(r); CHKERRA(ierr);
  ierr = VecDestroy(u); CHKERRA(ierr);
  ierr = MatDestroy(A); CHKERRA(ierr);
  ierr = SNESDestroy(snes); CHKERRA(ierr);
  ierr = GridDestroy(grid);

  PetscFinalize();
  return 0;
}

#include <math.h>

/* -------------------------------------------------------------------- */
/*
   SolutionFunction - Defines the solution to the PDE. It evaluates the 
   solution at any set of points.

   Input Parameters:
.  n - array dimension
.  x,y,z - grid point coordinates (not used here)
.  values - array to hold solution values
.  ctx - optional user-defined context (not used here)

   Output Parameter:
.  values - newly computed solution values
*/
int SolutionFunction(int n,double *x,double *y,double *z,Scalar *values,void *ctx)
{
  int i;

  for ( i=0; i<n; i++ ) values[i] = sqrt(x[i]);
  return 0;
}
/* -------------------------------------------------------------------- */
/*
   FormFunction - Evaluates nonlinear function, F(x).

   Input Parameters:
.  snes - the SNES context
.  x - input vector
.  ctx - optional user-defined context, as set by SNESSetFunction()

   Output Parameter:
.  f - function vector
*/
int FormFunction(SNES snes,GVec x,Vec f,void* ctx)
{
  int     ierr;

  ierr = UserApplyOperator(x,f); CHKERRQ(ierr);
  ierr = UserApplyDirichlet(x,f,SolutionFunction,0); CHKERRQ(ierr);
  return 0;
} 
/* -------------------------------------------------------------------- */
/*
   FormJacobian - Evaluates Jacobian matrix, F'(x).

   Input Parameters:
.  snes - the SNES context
.  x - input vector
.  ctx - optional user-defined context, as set by SNESSetJacobian()

   Output Parameters:
.  A - Jacobian matrix
.  B - optionally different preconditioning matrix
.  str - flag indicating matrix structure

*/
int FormJacobian(SNES snes,GVec x,GMat *A,GMat *B,MatStructure* str,void* ctx)
{
  int     ierr;

  ierr = UserComputeMatrix(x,*B); CHKERRQ(ierr);
  ierr = UserComputeDirichlet(x,*B,SolutionFunction,0); CHKERRQ(ierr);
  return 0;
}



