#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: ex11.c,v 1.3 1998/04/13 18:03:02 bsmith Exp $";
#endif

/*T
    Concepts: Grid vectors
    Routines: GridCreateRectangular1D(); GridCreateGVec();
    Routines: ViewerDrawOpenX(); GridDestroy(); GridCreateGMat();
    Routines: GVecEvaluateFunctionGalerkin();
    Routines: SLESSetFromOptions(); SLESSolve(); SLESSetFromOptions();
    Routines: KSPBuildSolution(); 

    Comment: Shows how to set up and solve a simple PDE using the GVec routines.
             In one dimension using quadratic discretization.
T*/

static char help[] = "Solves simple PDE with GVec routines.\n\n";

#include "sles.h"
#include "gvec.h"

int SolutionFunction(int,double *,double *,double *,Scalar *,void*);
int RHSFunction(int,double *,double *,double *,Scalar *,void*);

#include "src/gvec/examples/tutorials/common1and2.c"

int main(int argc,char **args)
{
  GVec                   x,b,u;                   /* discrete grid vectors */
  GMat                   A;
  Grid                   grid;
  int                    ierr, n = 10, flag, its;
  SLES                   sles;                    /* linear solver context */
  KSP                    ksp;
  Draw                   draw;
  GVecErrorKSPMonitorCtx monitorctx;

  PetscInitialize(&argc,&args,0,help);
  ierr = GVecInitialize();  CHKERRA(ierr);

  ierr = OptionsGetInt(PETSC_NULL,"-n",&n,&flag); CHKERRA(ierr);

  /*
      Construct the grid and discretization context
  */
  ierr = GridCreateRectangular1D(PETSC_COMM_WORLD,n,0.0,1.0,PETSC_NULL,DISCRETIZATION_QUADRATIC,&grid);
         CHKERRA(ierr);

  /* 
      From it create the grid vector and destroy the no longer needed Grid
  */
  ierr = GridCreateGVec(grid,&x); CHKERRA(ierr);

  /*
      Create another vector to hold the right hand side and exact solution
  */
  ierr = VecDuplicate(x,&b); CHKERRA(ierr);
  ierr = VecDuplicate(x,&u); CHKERRA(ierr);

  /*
      Evaluate the right hand side function on the grid. For finite
    differences this corresponds to the right hand side vector for 
    the linear system.
  */
  ierr = GVecEvaluateFunctionGalerkin(b,RHSFunction,0); CHKERRA(ierr);

  /*
     Compute the stiffness matrix for the Laplacian
  */
  ierr = GridCreateGMat(grid,&A); CHKERRA(ierr); 
  ierr = UserComputeMatrix(A); CHKERRA(ierr);

  /*
     Apply the Dirichlet boundary conditions to the vectors and 
     the matrix.
  */
  ierr = UserComputeDirichlet(x,b,A,SolutionFunction,0);CHKERRA(ierr);

  /*
     Solve the linear system to compute the solution to the 
     finite difference equations.
  */
  ierr = SLESCreate(PETSC_COMM_WORLD,&sles); CHKERRA(ierr);

  /*
     Set convergence monitoring routine to plot the error
  */
  ierr = SLESGetKSP(sles,&ksp); CHKERRA(ierr);
  ierr = ViewerDrawGetDraw(VIEWER_DRAWX_WORLD,&draw); CHKERRA(ierr);
  ierr = DrawSetTitle(draw,"Error"); CHKERRA(ierr);
  ierr = GVecEvaluateFunction(u,SolutionFunction,0); CHKERRA(ierr);
  monitorctx.error_viewer      = VIEWER_DRAWX_WORLD;
  monitorctx.solution          = u;
  monitorctx.norm_error_viewer = VIEWER_STDOUT_WORLD;
  ierr = KSPSetMonitor(ksp,GVecErrorKSPMonitor,&monitorctx);CHKERRA(ierr);
  
  /*
     Set the matrix in the linear system
  */
  ierr = SLESSetOperators(sles,A,A,SAME_NONZERO_PATTERN); CHKERRA(ierr);
  ierr = SLESSetFromOptions(sles); CHKERRA(ierr);
  ierr = SLESSolve(sles,b,x,&its); CHKERRA(ierr);

  ierr = VecDestroy(x); CHKERRA(ierr);
  ierr = VecDestroy(b); CHKERRA(ierr);
  ierr = VecDestroy(u); CHKERRA(ierr);
  ierr = MatDestroy(A); CHKERRA(ierr);
  ierr = SLESDestroy(sles); CHKERRA(ierr);
  ierr = GridDestroy(grid); CHKERRA(ierr);

  PetscFinalize();
  return 0;
}

/*
       This defines the solution to the PDE. It evaluates the 
   solution at any set of points
*/
int SolutionFunction(int n,double *x,double *y,double *z,Scalar *values,void *ctx)
{
  int i;

  for ( i=0; i<n; i++ ) values[i] = .5*x[i]*x[i];
  return 0;
}

/*
       This defines the right hand side function for this problem.
*/
int RHSFunction(int n,double *x,double *y,double *z,Scalar *values,void *ctx)
{
  int i;

  for ( i=0; i<n; i++ ) values[i] = 1.0;
  return 0;
}
