#ifdef PETSC_RCS_HEADER
static char vcid[] = "$Id: ex2.c,v 1.41 1998/03/23 21:24:16 bsmith Exp $";
#endif

static char help[] = "Tests some utilities for vectors with the extra property\n\
of being discrete functions (case of regular grid using multiple processors).\n\
Use the option -info_view to print information about the grid.\n\n";

#include "vec.h"
#include "da.h"
#include "dfvec.h"
#include <math.h>

int ViewCoordinates(MPI_Comm,double*,double*,double*,int,int,int);

int main(int argc,char **args)
{
  DFVec   v, lv;        /* discrete function vectors - initial global and local */
  DFVec   rv, rlv;      /* discrete function vector - refined global and local */
  DFVec   rv_work;      /* discrete function vector - used for intermediate work */
  DF      df;           /* discrete function shell - global */
  DF      rdf;          /* discrete function shell - refined global */
  Vec     *vsub;        /* standard vectors */
  DA      da, rda;      /* distributed arrays - initial and refined */
  Viewer  viewer1;
  Scalar  *lva;
  double  *x, *y, *z, *xx, *yy, *zz, ten = 10.0;
  int     i, j, nc = 2, mx = 3, my = 4, mz = 1, Nglob, flg, size;
  int     width = 300, height = 300, ierr, Nloc, rank, k, l, row, Nx, Ny, Nz;
  int     xs, xe, ys, ye, zs, ze, xm, ym, zm, Xs, Ys, Zs, Xm, Ym, Zm, nsub;
  int     contours = 0, info_view = 0;

  PetscInitialize(&argc,&args,0,help);
  ierr = ViewerSetFormat(VIEWER_STDOUT_WORLD,VIEWER_FORMAT_ASCII_COMMON,0); CHKERRA(ierr);

  /* Set processor configuration */
  MPI_Comm_size(PETSC_COMM_WORLD,&size);
  MPI_Comm_rank(PETSC_COMM_WORLD,&rank);
  Nx = 1; Ny = size; Nz = 1;
  ierr = OptionsGetInt(PETSC_NULL,"-Nx",&Nx,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-Ny",&Ny,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-Nz",&Nz,&flg); CHKERRA(ierr);
  /* really need better test here */
  if (Nx*Ny*Nz != size && (Nx != PETSC_DECIDE || Ny != PETSC_DECIDE || Nz != PETSC_DECIDE))
    SETERRA(1,0,"Incompatible number of processors:  Nx * Ny * Nz != size");
  ierr = OptionsGetInt(PETSC_NULL,"-mx",&mx,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-my",&my,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-mz",&mz,&flg); CHKERRA(ierr);
  ierr = OptionsGetInt(PETSC_NULL,"-nc",&nc,&flg); CHKERRA(ierr);
  if (nc<1 || mx<1 || my<1 || mz<1) SETERRA(1,0,"Only mx, my, mz, nc > 0 supported");
  ierr = OptionsHasName(PETSC_NULL,"-info_view",&info_view); CHKERRA(ierr);
  ierr = OptionsHasName(PETSC_NULL,"-contours",&contours); CHKERRA(ierr);

  /* Create distributed array, global and local vectors */
  if (my == 1 && mz == 1) {
    ierr = DACreate1d(PETSC_COMM_WORLD,DA_NONPERIODIC,mx,nc,1,PETSC_NULL,&da); CHKERRA(ierr);
  } 
  else if (mz == 1) {
    ierr = DACreate2d(PETSC_COMM_WORLD,DA_NONPERIODIC,DA_STENCIL_STAR,mx,
                      my,Nx,Ny,nc,1,PETSC_NULL,PETSC_NULL,&da); CHKERRA(ierr);
  }
  else {
    ierr = DACreate3d(PETSC_COMM_WORLD,DA_NONPERIODIC,DA_STENCIL_STAR,mx,
                      my,mz,Nx,Ny,Nz,nc,1,PETSC_NULL,PETSC_NULL,PETSC_NULL,&da);CHKERRA(ierr);
  }
  if (info_view) {
    ierr = DAView(da,VIEWER_STDOUT_SELF); CHKERRA(ierr);
    ierr = ViewerDrawOpenX(PETSC_COMM_WORLD,0,"",300,0,400,400,&viewer1); CHKERRA(ierr);
    ierr = DAView(da,viewer1); CHKERRA(ierr);
  }

  ierr = DACreateGlobalVector(da,&v); CHKERRA(ierr);
  ierr = DACreateLocalVector(da,&lv); CHKERRA(ierr);

  /* Determine locally owned region */
  ierr = DAGetCorners(da,&xs,&ys,&zs,&xm,&ym,&zm); CHKERRA(ierr);
  ierr = DAGetGhostCorners(da,&Xs,&Ys,&Zs,&Xm,&Ym,&Zm); CHKERRA(ierr);
  ierr = VecGetSize(v,&Nglob); CHKERRA(ierr);
  if (ym == 0) ym = 1; if (zm == 0) zm = 1;  /* fix for 1D and 2D problems */
  Nloc = xm*ym*zm*nc;

  if (info_view) {
    PetscPrintf(PETSC_COMM_WORLD,
      "Nx=%d, Ny=%d, Nz=%d, size=%d, mx=%d, my=%d, mz=%d, Nglob=%d\n",
      Nx,Ny,Nz,size,mx,my,mz,Nglob);
    PetscSequentialPhaseBegin(PETSC_COMM_WORLD,1);
    printf("[%d] xs=%d, xm=%d, ys=%d, ym=%d, zs=%d, zm=%d, Nloc=%d\n",
      rank,xs,xm,ys,ym,zs,zm,Nloc); fflush(stdout);
    printf("[%d] Xs=%d, Xm=%d, Ys=%d, Ym=%d, Zs=%d, Zm=%d\n",
      rank,Xs,Xm,Ys,Ym,Zs,Zm); fflush(stdout);
    PetscSequentialPhaseEnd(PETSC_COMM_WORLD,1);
  }
  x = (double *) PetscMalloc( mx*sizeof(double) ); CHKPTRA(x);
  y = (double *) PetscMalloc( my*sizeof(double) ); CHKPTRA(y);
  z = (double *) PetscMalloc( mz*sizeof(double) ); CHKPTRA(z);
  xe = xs+xm; ye = ys+ym; ze = zs+zm;

  /* Set vector values */
  ierr = VecGetArray(lv,&lva); CHKERRA(ierr);
  for (l=1; l<nc+1; l++) {
    for (k=zs; k<ze; k++) {
      for (j=ys; j<ye; j++) {
        for (i=xs; i<xe; i++) {
          row = i-Xs + (j-Ys)*Xm + (k-Zs)*Ym*Xm;
          lva[row*nc+l-1] = k*pow(ten,(double)(2*l)) + j*pow(ten,(double)(2*l-1)) + i;
        }
      }
    }
  }
  ierr = VecRestoreArray(lv,&lva); CHKERRA(ierr);

  /* Set grid points */
  for (k=0; k<mz; k++) z[k] = k*0.5;
  for (j=0; j<my; j++) y[j] = j*0.4;
  for (i=0; i<mx; i++) x[i] = i*0.3;

  /* Insert values into global vector */
  ierr = DALocalToGlobal(da,lv,INSERT_VALUES,v); CHKERRA(ierr);

  /* Create vectors for separate components */
  if (info_view) {
    ierr = ViewerPushFormat(VIEWER_STDOUT_WORLD,VIEWER_FORMAT_ASCII_INFO,
           PETSC_NULL); CHKERRA(ierr);
    ierr = VecView(v,VIEWER_STDOUT_WORLD); CHKERRA(ierr);
    ierr = ViewerPopFormat(VIEWER_STDOUT_WORLD); CHKERRA(ierr);
  }
  ierr = DFVecGetComponentVectors(lv,&nsub,&vsub); CHKERRA(ierr);

  /* Apply some user-defined function to vector vsub[0] */
  ierr = VecGetArray(vsub[0],&lva); CHKERRA(ierr);
  for (k=zs; k<ze; k++) {
    for (j=ys; j<ye; j++) {
      for (i=xs; i<xe; i++) {
        row = i-Xs + (j-Ys)*Xm + (k-Zs)*Ym*Xm;
        lva[row] += 80000;
      }
    }
  }
  ierr = VecRestoreArray(vsub[0],&lva); CHKERRA(ierr);
  ierr = DFVecAssembleFullVector(vsub,lv); CHKERRA(ierr);

  /* Insert values into global vector */
  ierr = DALocalToGlobal(da,lv,INSERT_VALUES,v); CHKERRA(ierr);

  /* View vectors */
  PetscPrintf(PETSC_COMM_WORLD,"multicomponent vector\n");
  ierr = DFVecView(v,VIEWER_STDOUT_WORLD); CHKERRA(ierr);
  /* for (i=0; i<nc; i++) {
    PetscPrintf(PETSC_COMM_WORLD,"component %d\n",i);
    ierr = VecView(vsub[i],VIEWER_STDOUT_WORLD); CHKERRA(ierr);
  } */

  /* Draw contour plot for each component */
  ierr = DFVecGetDFShell(v,&df); CHKERRA(ierr);
  ierr = DFShellSetCoordinates(df,mx,my,mz,x,y,z); CHKERRA(ierr);
  ierr = ViewCoordinates(PETSC_COMM_WORLD,x,y,z,mx,my,mz); CHKERRA(ierr);
  if (contours) {ierr = DFVecDrawTensorContoursX(v,width,height); CHKERRA(ierr);}

  /* Refine vector */
  /* Note:  Currently the vector returned in DFVecRefineVector()
            when using DA's is not a complete DFVec, so we also
            refine the DA, then copy the refined vector into the
            new DA-associated vector.  This is clearly not a good
            way to handle the refinement; we'll upgrade this in
            the near future. 
  */
  ierr = DFVecRefineVector(v,&rv_work); CHKERRA(ierr);

  /* Refine DA and copy vector to the new DA-associated vector */
  ierr = DARefine(da,&rda); CHKERRA(ierr);
  ierr = DACreateLocalVector(rda,&rlv); CHKERRA(ierr);
  ierr = DACreateGlobalVector(rda,&rv); CHKERRA(ierr);
  ierr = DFVecCopy(rv_work,rv); CHKERRA(ierr);


  if (info_view) {
    ierr = ViewerPushFormat(VIEWER_STDOUT_WORLD,VIEWER_FORMAT_ASCII_INFO,
           PETSC_NULL); CHKERRA(ierr);
    ierr = VecView(rv,VIEWER_STDOUT_WORLD); CHKERRA(ierr);
    ierr = ViewerPopFormat(VIEWER_STDOUT_WORLD); CHKERRA(ierr);
  }

  /* View refined vectors */
  PetscPrintf(PETSC_COMM_WORLD,"refined multicomponent vector\n");
  ierr = DFVecView(rv,VIEWER_STDOUT_WORLD); CHKERRA(ierr);
  ierr = VecDestroyVecs(vsub,nc); CHKERRA(ierr);
  ierr = DFVecGetComponentVectors(rv,&nsub,&vsub); CHKERRA(ierr);
  /* for (i=0; i<nc; i++) {
    PetscPrintf(PETSC_COMM_WORLD,"refined vector component %d\n",i);
    ierr = VecView(vsub[i],VIEWER_STDOUT_WORLD); CHKERRA(ierr);
  } */

  /* View refined grid coordinates */
  ierr = DFVecGetDFShell(rv,&rdf); CHKERRA(ierr);
  ierr = DFShellGetCoordinates(rdf,&mx,&my,&mz,&xx,&yy,&zz); CHKERRA(ierr);
  ierr = ViewCoordinates(PETSC_COMM_WORLD,xx,yy,zz,mx,my,mz); CHKERRA(ierr);
  if (contours) {ierr = DFVecDrawTensorContoursX(rv,width,height); CHKERRA(ierr);}

  /* Destroy data structures */
  PetscFree(x); PetscFree(y); PetscFree(z);
  ierr = VecDestroy(v); CHKERRA(ierr);  ierr = VecDestroy(lv); CHKERRA(ierr);
  ierr = VecDestroy(rv); CHKERRA(ierr); ierr = VecDestroy(rlv); CHKERRA(ierr); 

  /* Destroy the refined shell for work vector */
  ierr = VecDestroy(rv_work); CHKERRA(ierr); 

  ierr = VecDestroyVecs(vsub,nc); CHKERRA(ierr);

  ierr = DADestroy(da); CHKERRA(ierr);
  ierr = DADestroy(rda); CHKERRA(ierr);
  PLogFlops(1);

  PetscFinalize();
  return 0;
}
/* --------------------------------------------------------------- */
/* 
    ViewCoordinates - Prints grid coordinates.
 */
int ViewCoordinates(MPI_Comm comm,double *x,double *y,double *z,
                    int mx,int my, int mz)
{
  int i;

  PetscPrintf(comm,"x coordinates:\n");
  for (i=0; i<mx; i++) 
    PetscPrintf(comm,"x[%d] = %g\n",i,x[i]);
  PetscPrintf(comm,"y coordinates:\n");
  for (i=0; i<my; i++)
    PetscPrintf(comm,"y[%d] = %g\n",i,y[i]);
  PetscPrintf(comm,"z coordinates:\n");
  for (i=0; i<mz; i++)
    PetscPrintf(comm,"z[%d] = %g\n",i,z[i]);

  return 0;
}
